// Service Worker for Tulip Website - Updated with Enhanced Offline Page
const CACHE_NAME = 'tulip-v' + Date.now();
console.log('🚀 SW: Loading Service Worker v17 with enhanced offline page and dynamic data');
// We'll use inline HTML instead of external file
const OFFLINE_PAGE = null;

// Check if in development environment
const isDevelopment = () => {
  return self.location.hostname.includes('localhost') ||
    self.location.hostname.includes('.test') ||
    self.location.hostname.includes('127.0.0.1');
};

// Development-only logging
const devLog = (message) => {
  if (isDevelopment()) {
    console.log(message);
  }
};

// Essential files to cache - no external offline page needed
const STATIC_CACHE_URLS = [
  // All offline content is now inline in the Service Worker
];

// Install event - simplified installation
self.addEventListener('install', event => {
  console.log('🚀 SW: Installing Service Worker v17 with inline offline page...');

  event.waitUntil(
    Promise.all([
      // Open main cache
      caches.open(CACHE_NAME)
        .then(() => {
          console.log('✅ SW: Cache opened successfully');
        }),
      // Cache offline data
      caches.open('offline-data-cache')
        .then((cache) => {
          console.log('📱 SW: Caching offline data...');
          return fetch('/api/offline-data')
            .then(response => {
              if (response.ok) {
                console.log('📱 SW: Successfully fetched offline data');
                return cache.put('/api/offline-data', response.clone());
              } else {
                console.log('📱 SW: Failed to fetch offline data, status:', response.status);
              }
            })
            .catch(error => {
              console.log('📱 SW: Could not fetch offline data:', error);
            });
        })
    ])
      .then(() => {
        console.log('✅ SW: Installation complete - offline page is inline with dynamic data');
        return self.skipWaiting();
      })
      .catch(error => {
        console.error('❌ SW: Install failed', error);
      })
  );
});

// Activate event - clean old caches and take control
self.addEventListener('activate', event => {
  devLog('🔄 SW: Activating...');

  event.waitUntil(
    caches.keys()
      .then(cacheNames => {
        return Promise.all(
          cacheNames.map(cacheName => {
            if (cacheName !== CACHE_NAME && cacheName !== 'offline-data-cache') {
              devLog('🗑️ SW: Deleting old cache:', cacheName);
              return caches.delete(cacheName);
            }
          })
        );
      })
      .then(() => {
        devLog('✅ SW: Activated and controlling all pages');
        return self.clients.claim();
      })
  );
});

// Periodic update of offline data
self.addEventListener('message', event => {
  if (event.data && event.data.type === 'UPDATE_OFFLINE_DATA') {
    updateOfflineData();
  }
});

// Function to update offline data
async function updateOfflineData() {
  try {
    console.log('📱 SW: Updating offline data...');
    const response = await fetch('/api/offline-data');
    if (response.ok) {
      const cache = await caches.open('offline-data-cache');
      await cache.put('/api/offline-data', response.clone());
      console.log('📱 SW: Offline data updated successfully');
    }
  } catch (error) {
    console.log('📱 SW: Failed to update offline data:', error);
  }
}

// Fetch event - handle all network requests
self.addEventListener('fetch', event => {
  const { request } = event;

  // Skip non-GET requests
  if (request.method !== 'GET') return;

  // Skip external requests
  if (!request.url.startsWith(self.location.origin)) return;

  // Skip service worker file itself
  if (request.url.includes('/sw.js')) return;

  devLog('🔍 SW: Intercepting request: ' + new URL(request.url).pathname);

  event.respondWith(handleFetch(request));
});

// Main fetch handler
async function handleFetch(request) {
  const url = new URL(request.url);

  // For page requests, use Network First strategy for fresh data
  if (isPageRequest(request)) {
    devLog('🌐 SW: Page request for: ' + url.pathname);

    // Try network first to get fresh data
    try {
      devLog('🌐 SW: Fetching fresh page from network: ' + url.pathname);

      // Set a shorter timeout for network requests to fail fast when offline
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 3000); // 3 second timeout

      const networkResponse = await fetch(request, {
        signal: controller.signal
      });

      clearTimeout(timeoutId);

      if (networkResponse.ok) {
        devLog('✅ SW: Fresh page fetched, updating cache...');
        await cacheResponse(request, networkResponse.clone());
        return networkResponse;
      } else {
        devLog('⚠️ SW: Server error (status: ' + networkResponse.status + ')');

        // Handle different error types
        if (networkResponse.status >= 500 && networkResponse.status < 600) {
          // Server errors (500, 502, 503, etc.) - show the actual error
          devLog('🚨 SW: Server error detected, showing Laravel error page');
          return networkResponse;
        } else if (networkResponse.status === 404) {
          // 404 errors - try cache first, then show error
          devLog('🔍 SW: 404 error, checking cache first');
          const cachedResponse = await caches.match(request);
          if (cachedResponse) {
            devLog('✅ SW: Serving cached page for 404: ' + url.pathname);
            return cachedResponse;
          } else {
            devLog('📄 SW: No cached page, showing 404 error');
            return networkResponse;
          }
        } else {
          // Other client errors (401, 403, etc.) - show the error
          devLog('🚫 SW: Client error, showing error page');
          return networkResponse;
        }
      }
    } catch (error) {
      console.log('❌ SW: Network completely failed: ' + error.message);
      console.log('🔍 SW: Checking cache for: ' + url.pathname);

      // Try to serve from cache first before showing offline page
      const cachedResponse = await caches.match(request);
      if (cachedResponse) {
        console.log('✅ SW: Serving cached page while offline: ' + url.pathname);
        return cachedResponse;
      } else {
        console.log('📱 SW: No cached page found, showing ENHANCED offline page');
        console.log('🎯 SW: Request URL was: ' + request.url);
        console.log('🚀 SW: About to serve enhanced offline page with contact info');
        return getOfflinePage();
      }
    }
  }

  // For other resources (CSS, JS, images, etc.)
  try {
    // Try cache first
    const cachedResponse = await caches.match(request);
    if (cachedResponse) {
      devLog('📦 SW: Serving from cache:', url.pathname);
      return cachedResponse;
    }

    // Try network
    devLog('🌐 SW: Fetching from network:', url.pathname);
    const networkResponse = await fetch(request);

    // Cache successful responses
    if (networkResponse.ok) {
      await cacheResponse(request, networkResponse.clone());
    }

    return networkResponse;

  } catch (error) {
    devLog('❌ SW: Network failed for resource:', url.pathname);

    // Return error for resources
    return new Response('Offline', {
      status: 503,
      statusText: 'Service Unavailable'
    });
  }
}

// Cache successful responses
async function cacheResponse(request, response) {
  try {
    const cache = await caches.open(CACHE_NAME);

    // For HTML pages, ALWAYS cache regardless of headers (force caching)
    if (isPageRequest(request)) {
      // Read the response body
      const responseBody = await response.clone().text();

      // Create a clean response without problematic headers
      const cleanResponse = new Response(responseBody, {
        status: response.status,
        statusText: response.statusText,
        headers: {
          'Content-Type': 'text/html; charset=utf-8',
          'Cache-Control': 'max-age=3600' // Cache for 1 hour
        }
      });

      await cache.put(request, cleanResponse);
      devLog('💾 SW: Cached page (forced):', request.url);
      return;
    }

    // For other resources, check headers
    const vary = response.headers.get('vary');
    if (vary === '*') {
      devLog('⚠️ SW: Skipping resource cache due to Vary: *');
      return;
    }

    const cacheControl = response.headers.get('cache-control');
    if (cacheControl?.includes('no-store')) {
      devLog('⚠️ SW: Skipping resource cache due to no-store');
      return;
    }

    await cache.put(request, response);
    devLog('💾 SW: Cached:', request.url);

  } catch (error) {
    devLog('⚠️ SW: Cache failed:', error.message);
  }
}

// Check if request is for a page
function isPageRequest(request) {
  // Primary check: navigation requests
  if (request.mode === 'navigate') {
    return true;
  }

  // Secondary check: document destination
  if (request.destination === 'document') {
    return true;
  }

  // Check Accept header for HTML
  const acceptHeader = request.headers.get('accept');
  if (acceptHeader && acceptHeader.includes('text/html')) {
    return true;
  }

  // Check URL patterns for pages (no file extension or common page paths)
  const url = new URL(request.url);
  const pathname = url.pathname;

  // Skip API endpoints and assets
  if (pathname.includes('/api/') ||
    pathname.includes('/assets/') ||
    pathname.includes('/_debugbar/') ||
    pathname.includes('/sw.js')) {
    return false;
  }

  // Check for file extensions (not a page if it has common file extensions)
  const fileExtensions = /\.(js|css|png|jpg|jpeg|gif|svg|ico|woff|woff2|ttf|eot|json|xml|txt|pdf|webp|avif|bmp)$/i;
  if (fileExtensions.test(pathname)) {
    return false;
  }

  // If it's a GET request and doesn't match exclusions, treat as page
  return request.method === 'GET';
}

// Get offline page - serve inline HTML directly
async function getOfflinePage() {
  console.log('🎉 SW: *** ENHANCED OFFLINE PAGE FUNCTION CALLED ***');
  console.log('📱 SW: User is offline, serving enhanced offline page with contact info...');

  // Try to get dynamic data from cache first
  let offlineData = null;
  try {
    const cache = await caches.open('offline-data-cache');
    const response = await cache.match('/api/offline-data');
    if (response) {
      offlineData = await response.json();
      console.log('📱 SW: Using cached offline data:', offlineData);
      console.log('📱 SW: Phones count:', offlineData.phones ? offlineData.phones.length : 0);
    } else {
      console.log('📱 SW: No cached offline data found, trying to fetch fresh data...');
      // Try to fetch fresh data if online
      if (navigator.onLine) {
        try {
          const freshResponse = await fetch('/api/offline-data');
          if (freshResponse.ok) {
            offlineData = await freshResponse.json();
            // Cache the fresh data
            await cache.put('/api/offline-data', freshResponse.clone());
            console.log('📱 SW: Fetched and cached fresh offline data:', offlineData);
            console.log('📱 SW: Fresh phones count:', offlineData.phones ? offlineData.phones.length : 0);
          } else {
            console.log('📱 SW: Fresh data fetch failed, status:', freshResponse.status);
          }
        } catch (fetchError) {
          console.log('📱 SW: Could not fetch fresh data:', fetchError);
        }
      } else {
        console.log('📱 SW: Offline, cannot fetch fresh data');
      }
    }
  } catch (error) {
    console.log('📱 SW: Could not get offline data:', error);
  }

  console.log('📱 SW: Final offlineData:', offlineData);

  // Fallback offline response - enhanced version matching the Laravel page
  return new Response(`
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>You are offline - Tulip</title>
        <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
        <style>
            body {
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                margin: 0;
                padding: 0;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                min-height: 100vh;
                display: flex;
                align-items: center;
                justify-content: center;
                color: white;
            }

            .offline-container {
                text-align: center;
                padding: 2rem;
                max-width: 800px;
                width: 90%;
                background: rgba(255, 255, 255, 0.1);
                border-radius: 20px;
                box-shadow: 0 20px 40px rgba(0, 0, 0, 0.2);
                backdrop-filter: blur(10px);
            }

            .offline-icon {
                font-size: 4rem;
                margin-bottom: 1rem;
            }

            .offline-title {
                font-size: 2.5rem;
                margin-bottom: 1rem;
                font-weight: 700;
            }

            .offline-message {
                font-size: 1.2rem;
                margin-bottom: 2rem;
                opacity: 0.9;
                line-height: 1.6;
            }

            .offline-actions {
                display: flex;
                gap: 1rem;
                justify-content: center;
                flex-wrap: wrap;
                margin-bottom: 2rem;
            }

            .btn {
                padding: 12px 24px;
                border: none;
                border-radius: 50px;
                font-size: 1rem;
                font-weight: 600;
                cursor: pointer;
                transition: all 0.3s ease;
                text-decoration: none;
                display: inline-flex;
                align-items: center;
                gap: 8px;
                background: rgba(255, 255, 255, 0.2);
                color: white;
                border: 1px solid rgba(255, 255, 255, 0.3);
            }

            .btn:hover {
                transform: translateY(-2px);
                box-shadow: 0 8px 20px rgba(0, 0, 0, 0.3);
                background: rgba(255, 255, 255, 0.3);
            }

            .contact-section {
                background: rgba(255, 255, 255, 0.1);
                padding: 1.5rem;
                border-radius: 15px;
                margin: 2rem 0;
                text-align: left;
            }

            .contact-section h3 {
                margin-bottom: 1rem;
                color: #ffc107;
                text-align: center;
            }

            .contact-grid {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
                gap: 1rem;
                margin-top: 1rem;
            }

            .contact-item {
                background: rgba(255, 255, 255, 0.1);
                padding: 1rem;
                border-radius: 10px;
            }

            .contact-item h4 {
                color: #17a2b8;
                margin-bottom: 0.5rem;
                font-size: 0.9rem;
            }

            .contact-item a {
                color: white;
                text-decoration: none;
                display: block;
                margin: 0.2rem 0;
            }

            .contact-item a:hover {
                color: #ffc107;
            }

            .offline-pages {
                background: rgba(255, 255, 255, 0.1);
                padding: 1.5rem;
                border-radius: 15px;
                margin: 2rem 0;
            }

            .offline-pages h3 {
                margin-bottom: 1rem;
                color: #ffc107;
                text-align: center;
            }

            .pages-grid {
                display: flex;
                flex-wrap: wrap;
                gap: 0.5rem;
                justify-content: center;
            }

            .pages-grid a {
                color: #17a2b8;
                text-decoration: none;
                padding: 0.5rem 1rem;
                border-radius: 25px;
                background: rgba(255, 255, 255, 0.1);
                transition: all 0.3s ease;
                font-size: 0.9rem;
            }

            .pages-grid a:hover {
                background: rgba(255, 255, 255, 0.2);
                color: #fff;
            }

            .tip-section {
                background: rgba(255, 193, 7, 0.2);
                padding: 1rem;
                border-radius: 10px;
                margin-top: 2rem;
                border-left: 4px solid #ffc107;
            }

            .tip-section strong {
                color: #ffc107;
            }

            @media (max-width: 768px) {
                .offline-container {
                    padding: 1.5rem;
                    margin: 1rem;
                }

                .offline-title {
                    font-size: 2rem;
                }

                .offline-actions {
                    flex-direction: column;
                    align-items: center;
                }

                .contact-grid {
                    grid-template-columns: 1fr;
                }
            }
        </style>
    </head>
    <body>
        <div class="offline-container">
            <div class="offline-icon">📡</div>
            <h1 class="offline-title">You are offline</h1>
            <p class="offline-message">
                It seems you've lost your internet connection. Don't worry! Some pages are available offline, and you can access the saved content.
            </p>

            <div class="offline-actions">
                <button class="btn" onclick="window.location.reload()">
                    🔄 Try again
                </button>
                <a href="/" class="btn">
                    🏠 Return to Home
                </a>
                <button class="btn" onclick="history.back()">
                    ← Go Back
                </button>
            </div>

            <div class="contact-section">
                <h3>📞 Contact Us</h3>
                <div class="contact-grid">
                    ${offlineData && offlineData.phones ? offlineData.phones.map(phone => `
                        <div class="contact-item">
                            <h4>${phone.type === 'whatsapp' ? '📱' : '📞'} ${phone.name}</h4>
                            <a href="tel:${phone.phone}">${phone.phone}</a>
                            ${phone.email ? `<a href="mailto:${phone.email}">${phone.email}</a>` : ''}
                        </div>
                    `).join('') : `
                        <div class="contact-item">
                            <h4>📞 Sales Phone</h4>
                            <a href="tel:+966501234567">+966501234567</a>
                            <a href="mailto:sales@site.com">sales@site.com</a>
                        </div>
                        <div class="contact-item">
                            <h4>🛠️ Technical Support Phone</h4>
                            <a href="tel:+966507654321">+966507654321</a>
                            <a href="mailto:support@site.com">support@site.com</a>
                        </div>
                        <div class="contact-item">
                            <h4>🚨 Emergency Phone</h4>
                            <a href="tel:+966509876543">+966509876543</a>
                            <a href="mailto:emergency@site.com">emergency@site.com</a>
                        </div>
                        <div class="contact-item">
                            <h4>👔 Management Phone</h4>
                            <a href="tel:+966501112223">+966501112223</a>
                            <a href="mailto:admin@site.com">admin@site.com</a>
                        </div>
                    `}
                </div>
            </div>

            <div class="offline-pages">
                <h3>📱 All pages available offline</h3>
                <div class="pages-grid">
                    <a href="/">🏠 Home Page</a>
                    <a href="/about-us">ℹ️ About Us</a>
                    <a href="/services">🛠️ Services</a>
                    <a href="/contact-us">📞 Contact Us</a>
                    <a href="/amp/home">⚡ Home Page AMP</a>
                </div>
            </div>

            <div class="tip-section">
                <strong>💡 Tip:</strong> 🔔 Enable notifications to get updates when you're back online!
            </div>
        </div>

        <script>
            // Auto-reload when connection is restored
            window.addEventListener('online', async () => {
                console.log('🌐 Connection restored! Updating offline data...');
                
                // Try to update offline data cache
                try {
                    const response = await fetch('/api/offline-data');
                    if (response.ok) {
                        const cache = await caches.open('offline-data-cache');
                        await cache.put('/api/offline-data', response.clone());
                        console.log('📱 Updated offline data cache with fresh data');
                    }
                } catch (error) {
                    console.log('📱 Could not update offline data:', error);
                }
                
                // Reload page to show updated data
                setTimeout(() => {
                    window.location.reload();
                }, 1000);
            });

            // Log offline event for analytics
            console.log('📱 User is viewing offline page');
            
            // Show online/offline status
            function updateConnectionStatus() {
                if (navigator.onLine) {
                    console.log('🌐 Online');
                } else {
                    console.log('📡 Offline');
                }
            }
            
            window.addEventListener('online', updateConnectionStatus);
            window.addEventListener('offline', updateConnectionStatus);
            updateConnectionStatus();
        </script>
    </body>
    </html>
  `, {
    headers: { 'Content-Type': 'text/html; charset=utf-8' }
  });
}

// Handle messages from main thread
self.addEventListener('message', event => {
  if (event.data?.type === 'SKIP_WAITING') {
    self.skipWaiting();
  } else if (event.data?.type === 'REFRESH_CACHE') {
    // Force refresh specific page cache
    const url = event.data.url;
    refreshPageCache(url);
  } else if (event.data?.type === 'CLAIM_CLIENTS') {
    // Force claim all clients immediately
    devLog('🔄 SW: Claiming all clients...');
    self.clients.claim();
  }
});

// Function to refresh page cache
async function refreshPageCache(url) {
  try {
    devLog('🔄 SW: Refreshing cache for:', url);
    const request = new Request(url);
    const response = await fetch(request);

    if (response.ok) {
      await cacheResponse(request, response.clone());
      devLog('✅ SW: Cache refreshed for:', url);
    }
  } catch (error) {
    devLog('⚠️ SW: Failed to refresh cache:', error.message);
  }
}

// Periodic cache refresh (every 30 minutes)
setInterval(async () => {
  try {
    const cache = await caches.open(CACHE_NAME);
    const requests = await cache.keys();

    // Only refresh HTML pages, not resources
    const pageRequests = requests.filter(req =>
      req.url.includes('/en') ||
      req.url.includes('/ar') ||
      req.url.endsWith('/')
    );

    devLog(`🔄 SW: Background refresh for ${pageRequests.length} pages`);

    // Refresh up to 5 pages to avoid overwhelming
    for (const request of pageRequests.slice(0, 5)) {
      try {
        const response = await fetch(request);
        if (response.ok) {
          await cacheResponse(request, response.clone());
          devLog('🔄 SW: Background updated:', request.url);
        }
      } catch (error) {
        // Silently fail for background updates
      }
    }
  } catch (error) {
    devLog('⚠️ SW: Background refresh failed:', error.message);
  }
}, 30 * 60 * 1000); // 30 minutes

devLog('✅ SW: Script loaded');
