<?php

namespace App\Services\ProductServices;

use App\Helpers\Media;
use App\Helpers\SaveImageTo3Path;
use App\Http\Requests\Admin\Products\UpdateProductRequest;
use App\Models\Product;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
class ProductService
{
    public function store($request)
    {

        DB::beginTransaction();
        try {
            $data = $request->validated();

            $data['link_en'] = Str::slug( $data['link_en']);
            $data['link_ar'] = preg_replace('/\s+/u', '-', trim($data['link_ar']));
          
            if ($request->hasFile('main_image')) {
                $data['main_image'] = Media::uploadAndAttachImage($request->file('main_image'), 'products');
            }
    
            if ($request->hasFile('second_image')) {
                $data['second_image'] = Media::uploadAndAttachImage($request->file('second_image'), 'products');
            }

            $product = Product::create($data);

            foreach ($data['attributes'] ??  [] as  $value) {
                $product->attributes()->create(['attribute_id' => $value]);
            }
    
            
            DB::commit();

            return $product;
        } catch (\Exception $e) {
            DB::rollBack();

            throw $e;
        }
    }

    public function update(UpdateProductRequest $request, Product $product)
    {
        DB::beginTransaction();
        
        try {
            // Prepare data with default values
            $data = $request->validated();
       
            $data['status'] = $data['status'] ?? 0;
            $data['featured'] = $data['featured'] ?? 0;
            $data['home'] = $data['home'] ?? 0;
            $data['link_en'] = Str::slug( $data['link_en']);
            $data['link_ar'] = preg_replace('/\s+/u', '-', trim($data['link_ar']));

            // Handle image updates
            if ($request->hasFile('main_image')) {
                if ($product->main_image) {
                    Media::removeFile('products', $product->main_image);
                }
                $data['main_image'] = Media::uploadAndAttachImage($request->file('main_image'), 'products');
            }
    
            if ($request->hasFile('second_image')) {
                if ($product->second_image) {
                    Media::removeFile('products', $product->second_image);
                }
                $data['second_image'] = Media::uploadAndAttachImage($request->file('second_image'), 'products');
            }
    
            // Handle attributes
            $currentAttributes = $product->attributes->pluck('attribute_id')->toArray();
            $newAttributes = $data['attributes'] ?? [];
            $hasNewAttributes = !empty(array_diff($newAttributes, $currentAttributes));
    
            $product->update($data);
    
            // Sync attributes
            $attributesToDelete = array_diff($currentAttributes, $newAttributes);
            $attributesToAdd = array_diff($newAttributes, $currentAttributes);
    
            if (!empty($attributesToDelete)) {
                $product->attributes()->whereIn('attribute_id', $attributesToDelete)->delete();
            }
    
            foreach ($attributesToAdd as $attributeId) {
                $product->attributes()->create(['attribute_id' => $attributeId]);
            }
    
            DB::commit();
            
            return $hasNewAttributes;
                
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }


    public function delete($selectedIds)
    {
        $products = Product::whereIn('id', $selectedIds)->get();

        DB::beginTransaction();
        try {
            foreach ($products as $product) {
                // Delete associated image if it exists
                if ($product->main_image) {
                    Media::removeFile('products', $product->main_image);
                }
                // Delete associated Icon if it exists
                if ($product->second_image) {
                    Media::removeFile('products', $product->second_image);
                }
            }
            $deleted = Product::whereIn('id', $selectedIds)->delete();

            DB::commit();

            return $deleted > 0;
        } catch (\Exception $e) {

            DB::rollBack();

            throw $e;
        }
    }

   
}
