<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class SimpleRequestOrder extends Model
{
    use HasFactory;

    protected $table = 'simple_request_orders';

    protected $fillable = [
        'order_number',
        'user_id',
        'address_id',
        'status_id',
        'shipping_fees',
        'note',
        'admin_seen',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($order) {

            $order->order_number = self::generateOrderNumber();
        });
    }

    public static function generateOrderNumber()
    {
        $latestOrder = self::orderBy('created_at', 'desc')->first();
        if ($latestOrder) {
            $lastOrderNumber = (int) substr($latestOrder->order_number, -6);
        } else {
            $lastOrderNumber = 0;
        }
        $newOrderNumber = str_pad($lastOrderNumber + 1, 6, '0', STR_PAD_LEFT);
        return config('site_prefix') . date('Ymd') . '-' . $newOrderNumber;
    }

    public function getTotalAmountAttribute()
    {
        return $this->shipping_fees;
    }
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function items()
    {
        return $this->hasMany(SimpleRequestOrderOrderItems::class, 'order_id');
    }

    public function address()
    {
        return $this->belongsTo(UserAddress::class, 'address_id');
    }

    public function status()
    {
        return $this->belongsTo(OrderStatus::class, 'status_id');
    }

    public function getStatusNameAttribute()
    {
        return $this->status?->name;
    }

    public function payment()
    {
        return $this->hasOne(Payment::class );
    }

   // This should be hasOne or hasMany for shipments, not using shipment_status_id
   public function shipments()
   {
       return $this->hasMany(OrderShipment::class, 'simple_request_order_id');
   }

   // Get the latest shipment
   public function latestShipment()
   {
       return $this->hasOne(OrderShipment::class, 'simple_request_order_id')->latest();
   }
}
