<?php

namespace App\Models;

use App\Models\OrderProduct;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;

class Order extends Model
{
    //
    protected $table = 'orders';

    protected $fillable = [
        'order_number',
        'user_id',
        'address_id',
        'status_id',
        'shipping_fees',
        'total_amount',
        'note',
        'admin_seen',
    ];


    protected static function boot()
    {
        parent::boot();

        static::creating(function ($order) {

            $order->order_number = self::generateOrderNumber();
        });
    }

    public static function generateOrderNumber()
    {
        $latestOrder = self::orderBy('created_at', 'desc')->first();
        if ($latestOrder) {
            $lastOrderNumber = (int) substr($latestOrder->order_number, -6);
        } else {
            $lastOrderNumber = 0;
        }
        $newOrderNumber = str_pad($lastOrderNumber + 1, 6, '0', STR_PAD_LEFT);
        return config('site_prefix') . date('Ymd') . '-' . $newOrderNumber;
    }

    public function items()
    {
        return $this->hasMany(OrderItem::class);
    }

    public function status()
    {
        return $this->belongsTo(OrderStatus::class, 'status_id');
    }

    public function getStatusNameAttribute()
    {
        return $this->status?->name;
    }

    public function orderShipments()
    {
        return $this->hasMany(OrderShipment::class);
    }

    public function address()
    {
        return $this->belongsTo(UserAddress::class);
    }


    public function payment()
    {
        return $this->hasOne(Payment::class);
    }

    public function totalProductsPrice()
    {
        return $this->items->sum(function ($item) {
            return $item->price * $item->quantity;
        });
    }

    public function user()
    {
        return $this->belongsTo('App\Models\User', 'user_id');
    }

    public static function orderCount()
    {
        return Order::where('admin_seen', 0)->count();
    }

    public function getTotalOrdersByPhoneAttribute()
    {
        $phone = $this->address?->phone;

        if (!$phone) {
            return 0;
        }

        return Order::whereHas('address', function ($q) use ($phone) {
            $q->where('phone', $phone);
        })->count();
    }

    public function getIsFirstOrderAttribute()
    {
        return $this->total_orders_by_phone === 1;
    }

    // filter //
    public function scopeByDateRange($query,$start_date = null, $end_date = null) {
        
        if($start_date && $end_date)
        {
            $query->whereBetween('created_at' , [Carbon::parse($start_date) ,Carbon::parse($end_date)]);
        }
    
        return $query;
    }

    public function scopeByAmountRange($query, $min_amount = null, $max_amount = null)
    {
     
        if ($min_amount && $max_amount)
        {
            $query->whereBetween('total_amount', [$min_amount, $max_amount]);
        }
        return $query;
    } 

    public function scopeByStatus($query, $status_id = null)
    {
        if ($status_id) {
            $query->where('status_id', $status_id);
        }
        return $query;
    }

    public function scopeByPaymentMethod($query , $payment_id = null)
    {
        if($payment_id)
        {
            $query->whereHas('payment', function($q) use ($payment_id){
                $q->where('payment_method_id', $payment_id);
            });
            return $query;
        }
    }

  
}
