<?php
namespace App\Models;

use App\Helpers\Helper;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Coupon extends Model
{
    protected $table = 'coupons';

    private $lang;

    protected $fillable = [
        'name_ar',
        'name_en',
        'code',
        'start_date',
        'end_date',
        'max_use',
        'min_price',
        'discount',
        'discount_type',
        'type',
    ];

    public function __construct(array $attributes = [])
    {
        parent::__construct($attributes);


        // Set the current locale dynamically
        $this->lang = Helper::getLang();
    }

    public function delete()
    {
        $errors = [];
        if ($this->users()->exists()) {
            $errors[] = 'The used coupon cannot be deleted.';
        }

        if (count($errors)) {
            return $errors;
        }

        return parent::delete();
    }

    protected $casts = [
        'start_date' => 'date:Y-m-d',
        'end_date' => 'date:Y-m-d',
        'discount' => 'float'
    ];

    public function getNameAttribute()
    {
        return $this->{'name_' . $this->lang};
    }

    public function users()
    {
        return $this->hasMany(CouponUsed::class);
    }

    public function getIsMaxUsedAttribute()
    {
        return $this->users->count() >= $this->max_use;
    }

    public function getIsStartAttribute()
    {
        $startDate = Carbon::parse($this->start_date);
        return $startDate->startOfDay() <= now()->startOfDay();
    }

    public function getIsEfficientAttribute()
    {
        $endDate = Carbon::parse($this->end_date);
        return $endDate->startOfDay() >= now()->startOfDay();
    }

    public function getIsUsedAttribute()
    {
        return $this->users->where('user_id', auth()->user()->id)->count() > 0;
    }

    public function checkMinPrice($total_price = 0): bool
    {
        return $this->min_price <= $total_price;
    }


    public function canUse($cart)
    {
        if (!$this->is_start) {
            return [false, __('home.Coupon is not yet active.')];
        }
    
        if (!$this->is_efficient) {
            return [false, __('home.Coupon has expired.')];
        }
    
        if (!$this->checkMinPrice($cart->products_price)) {
            return [false, __('home.cart_min_price_error') . ' ' . $this->min_price . ' EGP'];        }
    
        if ($this->is_max_used) {
            return [false, __('home.Coupon usage limit has been reached.')];
        }
    
        if ($this->is_used) {
            return [false, __('home.You have already used this coupon.')];
        }
    
        return [true, ''];
    }

    public function getDiscountTypeTextAttribute()
    {
        return $this->discount_type ? __('home.percentage') : __('home.amount');
    }

    public function getStartDateAttribute($value)
    {
        return Carbon::parse($value)->format('Y-m-d');
    }

    public function getEndDateAttribute($value)
    {
        return Carbon::parse($value)->format('Y-m-d');
    }

     // Calculate coupon amount based on the type of discount
     public function getCouponAmountAttribute()
     {
        if ($this->discount_type) {
            // Percentage discount
            return ($this->discount * 100) / 100;
        }
        
        // Fixed discount amount
        return $this->discount;
     }
}
