<?php

namespace App\Http\Controllers\Admin\Products;

use App\Helpers\Media;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Image;
use App\Models\ProductVariant;
use Illuminate\Support\Facades\DB;

class VariantImageController extends Controller
{
    public function index(ProductVariant $variant)
    {
        $data['variant'] = $variant;

        return view('admin.products.variants.images', $data);
    }

    public function store(Request $request, ProductVariant $variant)
    {
      
        DB::beginTransaction();
        try {
            // Validate the request
            $this->validate($request, [
                'images' => 'required|array',
                'images.*' => 'image|mimes:webp|max:1024',
            ]);

            // Process each uploaded image
            foreach ($request->file('images') as $file) {

                $filename = Media::uploadAndAttachImage($file, 'products/variants');
               
                Image::create([
                    'image' => $filename,
                    'imageable_id' => $variant->id,
                    'imageable_type' => $variant->getMorphClass(),
                ]);
            }
            DB::commit();

            return redirect()->route('variants.images.index', $variant->id)
                ->with('success', __('home.your_items_added_successfully'));

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors(__('home.an error occurred: ' . $e->getMessage()));
        }
    }

    public function destroy(Request $request, ProductVariant $variant, Image $image)
    {
        DB::beginTransaction();
        try {
            // Delete the image record
            $image->delete();

            // Remove the physical file
            Media::removeFile('products/variants', $image->image);

            DB::commit();
            return redirect()->back()->with('success', __('home.your_item_deleted_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors(__('home.an error occurred: ' . $e->getMessage()));
        }
    }
}
