<?php

namespace App\Http\Controllers\Admin;

use App\Helpers\Media;
use App\Helpers\SaveImageTo3Path;
use App\Http\Controllers\Controller;
use App\Http\Requests\CategoryRequest;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class CategoryController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */

    public function __construct()
    {
        $this->middleware(['permission:categories']);
    }

    public function index()
    {

        $categories = Category::with('parent')->orderBy('id', 'DESC')->get();
        return view('admin.categories.categories', compact('categories'));
    }


    public function create()
    {
        $categories = Category::where('status', 1)->get();
        return view('admin.categories.addCategory', compact('categories'));
    }


    public function store(CategoryRequest $request)
    {
        DB::beginTransaction();
        try {

            $data = $request->validated();

            $data['link_en'] = Str::slug($data['link_en']);

            $data['link_ar'] = preg_replace('/\s+/u', '-', trim($data['link_ar']));


            if ($request->hasFile('image')) {
                $data['image'] = Media::uploadAndAttachImage($request->file('image'), 'categories');
            }

            if ($request->hasFile('icon')) {
                $data['icon'] = Media::uploadAndAttachImage($request->file('icon'), 'categories');
            }

            Category::create($data);

            DB::commit();

            return redirect()->route('categories.index')->with('success', trans('home.your_item_added_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors($e->getMessage());
        }
    }


    public function edit(Category $category)
    {

        $categories = Category::where('id', '!=', $category->id)->get();
        return view('admin.categories.editCategory', compact('categories', 'category'));
    }


    public function update(CategoryRequest $request,  $category)
    {
        $category = Category::findOrFail($category);
        DB::beginTransaction();
        try {
            $data = $request->validated();

            $data['link_en'] = Str::slug($data['link_en']);
            $data['link_ar'] = preg_replace('/\s+/u', '-', trim($data['link_ar']));
            $data['status'] = $data['status'] ?? 0;

            if ($request->hasFile('image')) {
                if ($category->image) {
                    Media::removeFile('categories', $category->image);
                }
                $data['image'] = Media::uploadAndAttachImage($request->file('image'), 'categories');
            }

            if ($request->hasFile('icon')) {
                if ($category->icon) {
                    Media::removeFile('categories', $category->icon);
                }
                $data['icon'] = Media::uploadAndAttachImage($request->file('icon'), 'categories');
            }

            $category->update($data);

            DB::commit();

            return redirect()->back()->with('success', trans('home.your_item_updated_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors($e->getMessage());
        }
    }




    public function destroy(Request $request)
    {

        $selectedIds = $request->input('id');

        $categories = Category::whereIn('id', $selectedIds)->get();

        DB::beginTransaction();
        try {
            foreach ($categories as $category) {
                // Delete associated image if it exists
                if ($category->image) {
                    Media::removeFile('categories', $category->image);
                }
                // Delete associated Icon if it exists
                if ($category->icon) {
                    Media::removeFile('categories', $category->icon);
                }
            }
            Category::whereIn('id', $selectedIds)->delete();

            DB::commit();

            return response()->json(['success' => true, 'message' => __('home.your_items_deleted_successfully')]);
        } catch (\Exception $e) {

            DB::rollBack();
            return response()->json(['error' => true, 'message' => __('home.an error has occurred. Please contact the developer to resolve the issue')]);
        }
    }
}
