<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreBranchRequest;
use App\Http\Requests\UpdateBranchRequest;
use App\Models\Branch;
use App\Models\Governorate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class BranchController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $branches = Branch::with('governorate')->get(); // Fetch branches with their governorates
        return view('admin.branches.index', compact('branches'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $governorates = Governorate::all(); // Fetch all governorates for the dropdown
        return view('admin.branches.create', compact('governorates')); // Add view for creating a new branch
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreBranchRequest $request)
    {
        try {
            $branch = Branch::create($request->validated());
            return redirect()->route('branches.index')->with('success', __('home.branch_created_successfully'));
        } catch (\Exception $e) {
            return redirect()->back()->withErrors(['error' => __('home.error_creating_branch' . $e->getMessage())]);
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Branch $branch)
    {
        $governorates = Governorate::all(); // Fetch all governorates for the dropdown
        return view('admin.branches.edit', compact('branch', 'governorates')); // Add view for editing a branch
    }


    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateBranchRequest $request, Branch $branch)
    {
        try {
            $data = $request->validated();
            $data['status'] = $data['status'] ?? false; // Ensure status is set to false if not provided

            $branch->update($data);

            return redirect()->route('branches.index')->with('success', __('home.branch_updated_successfully'));
        } catch (\Exception $e) {
            return redirect()->back()->withErrors(['error' => __('home.error_updating_branch' . $e->getMessage())]);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request)
    {


        // Validate the request
        $request->validate([
            'id' => 'required|array|min:1',
            'id.*' => 'integer|exists:branches,id',
        ]);

        $selectedIds = $request->input('id');

        DB::beginTransaction();
        try {

            // Delete the branches
            $deleted = Branch::whereIn('id', $selectedIds)->delete(); // Updated to delete branches instead of inspection requests

            DB::commit();

            // Handle AJAX requests
            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => __('home.your_items_deleted_successfully'),
                ]);
            }

            // Redirect for non-AJAX requests
            return redirect()->route('home.contact_messages.index')
                ->with('success', __('home.your_items_deleted_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();

            // Log the error for debugging
            \Log::error('Error deleting contact messages: ' . $e->getMessage());

            // Handle AJAX requests
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => __('home.an_error_occurred'),
                ], 500);
            }

            // Redirect for non-AJAX requests
            return redirect()->back()
                ->withErrors(__('home.an_error_occurred'));
        }
    }
}
