<?php

namespace App\Http\Controllers;

use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class ManifestController extends Controller
{
    /**
     * Generate dynamic manifest.json
     */
    public function manifest(): JsonResponse
    {
        // Get current locale
        $currentLocale = app()->getLocale();

        // Get site settings from database (general settings)
        $settings = Setting::where('lang', 'all')->pluck('value', 'key')->toArray();

        // Get PWA settings for current language
        $pwaSettings = Setting::whereIn('key', ['site_short_name', 'site_description', 'theme_color', 'background_color', 'site_logo'])
            ->where('lang', $currentLocale)
            ->pluck('value', 'key')
            ->toArray();

        // Get site name and description from current locale settings
        $siteName = $pwaSettings['site_name'] ?? config("configrations.site_name") ?? ($currentLocale === 'ar' ? 'موقعي' : 'My Website');
        $siteDescription = $pwaSettings['site_description'] ?? config("configrations.site_description") ?? 'Professional web development, SEO, and digital marketing services in Egypt';
        $siteShortName = $pwaSettings['site_short_name'] ?? config("configrations.site_name") ?? 'Tulip';

        // Get theme colors
        $themeColor = $pwaSettings['theme_color'] ?? $settings['theme_color'] ?? '#007bff';
        $backgroundColor = $pwaSettings['background_color'] ?? $settings['background_color'] ?? '#ffffff';

        // Get site logo from current locale settings - prioritize current language
        $logoUrl = config("configrations.site_logo");

        // If no logo found in current locale, try to get from 'all' language
        if (!$logoUrl || $logoUrl === asset('assets/dashboard/images/noimage.png')) {
            $logoUrl = $settings['site_logo'] ?? null;
        }

        // If still no valid logo, use default favicon
        if (!$logoUrl || $logoUrl === asset('assets/dashboard/images/noimage.png')) {
            $logoUrl = asset('favicon.ico');
        }

        // Get current locale
        $currentLocale = app()->getLocale();
        $isRTL = $currentLocale === 'ar';

        // Generate shortcuts based on available pages/menus
        $shortcuts = $this->generateShortcuts($currentLocale);

        // Generate icons array
        $icons = $this->generateIcons($logoUrl);

        $manifest = [
            'name' => $siteName,
            'short_name' => $siteShortName,
            'description' => $siteDescription,
            'start_url' => '/' . $currentLocale,
            'display' => 'standalone',
            'background_color' => $backgroundColor,
            'theme_color' => $themeColor,
            'orientation' => 'portrait-primary',
            'icons' => $icons,
            'categories' => [
                'business',
                'productivity',
                'utilities'
            ],
            'lang' => $currentLocale,
            'dir' => $isRTL ? 'rtl' : 'ltr',
            'scope' => '/',
            'prefer_related_applications' => false,
            'related_applications' => [],
            'screenshots' => [],
            'shortcuts' => $shortcuts
        ];

        return response()->json($manifest)
            ->header('Content-Type', 'application/manifest+json')
            ->header('Cache-Control', 'no-cache, no-store, must-revalidate') // Force refresh
            ->header('Pragma', 'no-cache')
            ->header('Expires', '0');
    }

    /**
     * Generate shortcuts based on available menus/pages
     */
    private function generateShortcuts(string $locale): array
    {
        $shortcuts = [];

        // Home shortcut
        $shortcuts[] = [
            'name' => $locale === 'ar' ? 'الصفحة الرئيسية' : 'Home',
            'short_name' => $locale === 'ar' ? 'الرئيسية' : 'Home',
            'description' => $locale === 'ar' ? 'الصفحة الرئيسية للموقع' : 'Website Homepage',
            'url' => '/' . $locale,
            'icons' => [
                [
                    'src' => asset('favicon.ico'),
                    'sizes' => '16x16 32x32 48x48'
                ]
            ]
        ];

        // Services shortcut
        $shortcuts[] = [
            'name' => $locale === 'ar' ? 'الخدمات' : 'Services',
            'short_name' => $locale === 'ar' ? 'الخدمات' : 'Services',
            'description' => $locale === 'ar' ? 'خدماتنا المتاحة' : 'Our Services',
            'url' => '/' . $locale . '/services',
            'icons' => [
                [
                    'src' => asset('favicon.ico'),
                    'sizes' => '16x16 32x32 48x48'
                ]
            ]
        ];

        // Products shortcut
        $shortcuts[] = [
            'name' => $locale === 'ar' ? 'المنتجات' : 'Products',
            'short_name' => $locale === 'ar' ? 'المنتجات' : 'Products',
            'description' => $locale === 'ar' ? 'منتجاتنا المتاحة' : 'Our Products',
            'url' => '/' . $locale . '/products',
            'icons' => [
                [
                    'src' => asset('favicon.ico'),
                    'sizes' => '16x16 32x32 48x48'
                ]
            ]
        ];

        // Contact shortcut
        $shortcuts[] = [
            'name' => $locale === 'ar' ? 'اتصل بنا' : 'Contact Us',
            'short_name' => $locale === 'ar' ? 'اتصل بنا' : 'Contact',
            'description' => $locale === 'ar' ? 'تواصل معنا' : 'Get in Touch',
            'url' => '/' . $locale . '/contact',
            'icons' => [
                [
                    'src' => asset('favicon.ico'),
                    'sizes' => '16x16 32x32 48x48'
                ]
            ]
        ];

        return $shortcuts;
    }

    /**
     * Generate icons array based on available logo
     */
    private function generateIcons(string $logoUrl): array
    {
        $icons = [];

        // Determine file type based on URL
        $isFavicon = $logoUrl === asset('favicon.ico');
        $isSvg = str_ends_with($logoUrl, '.svg');
        $isPng = str_ends_with($logoUrl, '.png');
        $isJpg = str_ends_with($logoUrl, '.jpg') || str_ends_with($logoUrl, '.jpeg');

        // Determine appropriate MIME type
        $mimeType = 'image/png'; // Default
        if ($isSvg) {
            $mimeType = 'image/svg+xml';
        } elseif ($isJpg) {
            $mimeType = 'image/jpeg';
        }

        // Default favicon - use site logo if available
        $icons[] = [
            'src' => $isFavicon ? asset('favicon.ico') : $logoUrl,
            'sizes' => '16x16 32x32 48x48',
            'type' => $isFavicon ? 'image/x-icon' : $mimeType
        ];

        // If we have a custom logo, generate different sizes
        if (!$isFavicon) {

            // Generate different icon sizes - prioritize larger sizes for PWA
            $sizes = ['192x192', '512x512', '152x152', '72x72', '96x96', '144x144', '384x384'];

            foreach ($sizes as $size) {
                $icons[] = [
                    'src' => $logoUrl,
                    'sizes' => $size,
                    'type' => $mimeType,
                    'purpose' => 'any maskable'
                ];
            }

            // Add specific PWA icons for better install experience
            $pwaIcon = asset('pwa-icon.svg');

            // 192x192 for PWA install - primary icon
            $icons[] = [
                'src' => $pwaIcon,
                'sizes' => '192x192',
                'type' => 'image/svg+xml',
                'purpose' => 'any'
            ];

            // 512x512 for PWA install - primary icon
            $icons[] = [
                'src' => $pwaIcon,
                'sizes' => '512x512',
                'type' => 'image/svg+xml',
                'purpose' => 'any'
            ];
        }

        return $icons;
    }
}
