<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use App\Http\Requests\Dashboard\Album\DeleteAlbumRequest;
use App\Http\Requests\Dashboard\Album\StoreAlbumRequest;
use App\Http\Requests\Dashboard\Album\UpdateAlbumRequest;
use App\Models\Album;
use App\Services\Dashboard\AlbumService;
use Illuminate\Http\Request;

class AlbumController extends Controller
{

    protected $albumService;
    
    public function __construct(AlbumService $albumService)
    {
        $this->albumService = $albumService;
    }
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $this->authorize('albums.view');

        $albums = Album::orderBy('id','desc')->get();


        return view('Dashboard.Albums.index', compact('albums'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {

        $this->authorize('albums.create');
        $data['album'] = new Album();
        $data['albums'] = Album::get();
        return view('Dashboard.Albums.create', $data);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreAlbumRequest $request)
    {
        $this->authorize('albums.store');
        try {
            $data = $request->validated();
            $response = $this->albumService->store($request, $data);
            // إضافة الصور بعد إنشاء المشروع
            if ($response && $request->hasFile('album_images')) {
                $album = $response; // دالة store ترجع كائن المشروع الجديد
                $maxOrder = 0;
                foreach ($request->file('album_images') as $image) {
                    $filename = \App\Helper\Media::uploadAndAttachImage($image, 'albums');
                    \App\Models\AlbumImage::create([
                        'album_id' => $album->id,
                        'image' => $filename,
                        'order' => ++$maxOrder,
                    ]);
                }
            }
            if (!$response) {
                return redirect()->back()->with(['error' => __('dashboard.failed_to_create_item')]);
            }
            return redirect()->route('dashboard.albums.index')->with(['success' => __('dashboard.your_item_created_successfully')]);
        } catch (\Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Album $album)
    {
        $this->authorize('albums.edit');
        $data['album'] = $album;
        $data['albums'] = Album::get();

        return view('Dashboard.Albums.edit', $data);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateAlbumRequest $request, Album $album)
    {
        $this->authorize('albums.update');
        try {
            $data = $request->validated();
            $response = $this->albumService->update($request, $data, $album);
            // إضافة الصور الجديدة عند التحديث
            if ($request->hasFile('album_images')) {
                $maxOrder = \App\Models\AlbumImage::where('album_id', $album->id)->max('order') ?? 0;
                foreach ($request->file('album_images') as $image) {
                    $filename = \App\Helper\Media::uploadAndAttachImage($image, 'albums');
                    \App\Models\AlbumImage::create([
                        'album_id' => $album->id,
                        'image' => $filename,
                        'order' => ++$maxOrder,
                    ]);
                }
            }
            if (!$response) {
                return redirect()->back()->with(['error' => __('dashboard.failed_to_update_item')]);
            }
            return redirect()->route('dashboard.albums.index')->with(['success' => __('dashboard.your_item_updated_successfully')]);
        } catch (\Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     */
        public function destroy(DeleteAlbumRequest $request, string $id)
    {
        $this->authorize('albums.delete');

        $selectedIds = $request->input('selectedIds');

        $data = $request->validated();

        $deleted = $this->albumService->delete($selectedIds,$data);


        if (request()->ajax()) {
            if (!$deleted) {
                return response()->json(['message' => $deleted ?? __('dashboard.an messages.error entering data')], 422);
            }
            return response()->json(['success' => true, 'message' => __('dashboard.your_items_deleted_successfully')]);
        }
        if (!$deleted) {
            return redirect()->back()->withErrors($delete ?? __('dashboard.an error has occurred. Please contact the developer to resolve the issue'));
        }
    }

    /**
    * Change the category of a album.
     */
    public function changeCategory(Request $request, $id)
    {
        $this->authorize('albums.update');
        $request->validate([
            'category_id' => 'required|exists:categories,id',
        ]);
        $album = Album::findOrFail($id);
        $album->category_id = $request->input('category_id');
        $album->save();
        return redirect()->back()->with('success', __('dashboard.your_item_updated_successfully'));
    }
    
    /**
            * Reorder album images using drag and drop
     */
    public function reorderImages(Request $request)
    {
        $imageIds = $request->input('image_ids');
        if (!is_array($imageIds)) {
            return response()->json(['success' => false, 'message' => __('dashboard.invalid_data')], 400);
        }
        try {
            foreach ($imageIds as $order => $imageId) {
                \App\Models\AlbumImage::where('id', $imageId)->update(['order' => $order]);
            }
            return response()->json(['success' => true, 'message' => __('dashboard.images_reordered_successfully')]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => __('dashboard.error_reordering_images')], 500);
        }
    }
}
