<?php

namespace App\Providers;

use App\Models\Dashboard\Hosting;
use App\Models\Dashboard\Menu;
use App\Models\Slider;
use App\Models\Product;
use App\Models\Phone;
use App\Models\Service;
use App\Models\Setting;
use App\Models\SiteAddress;
use App\Models\Project;
use App\Models\Client;
use App\Models\Partener;
use App\Models\Section;
use App\Observers\BlogObserver;
use App\Observers\ProductObserver;
use App\Observers\ProjectObserver;
use App\Observers\ServiceObserver;
use App\Models\Blog;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Facades\View;
use Illuminate\Support\ServiceProvider;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        //
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        // Force HTTPS in production
        if($this->app->environment('production')) {
            URL::forceScheme('https');
            
            // Additional security headers for production
            if (request()->isSecure()) {
                header('Strict-Transport-Security: max-age=31536000; includeSubDomains');
                header('X-Content-Type-Options: nosniff');
                header('X-Frame-Options: DENY');
                header('X-XSS-Protection: 1; mode=block');
            }
        }

        // Set the application URL from config
        URL::forceRootUrl(config('app.url'));

        // Prevent lazy loading in non-production environments
        Model::preventLazyLoading(! $this->app->environment('production'));

        $lang = app()->getLocale();

        //  Check if the `settings` table exists before querying
        if (Schema::hasTable('settings')) {
            $configrations = Cache::remember("settings_{$lang}", 3600, function () use ($lang) {
                return Setting::where('lang', $lang)
                    ->get()
                    ->mapWithKeys(function ($item) {
                        return [$item->key => $item->value];
                    })->toArray();
            });

            $settings = Cache::remember("settings", 3600, function () {
                return Setting::where('lang', 'all')
                    ->get()
                    ->mapWithKeys(function ($item) {
                        return [$item->key => $item->value];
                    })->toArray();
            });

            Config::set('configrations', $configrations);
            Config::set('settings', $settings);
        }


        if (Schema::hasTable('menus')) {
            $menus = Menu::active()->get();
            View::share('menus', $menus);
        }

        if (Schema::hasTable('site_addresses')) {
            $site_addresses = SiteAddress::active()->first();
            View::share('site_addresses', $site_addresses);
        }

        if (Schema::hasTable('services')) {
            $headerServices = Service::header()->active()->take(4)->get();
            View::share('headerServices', $headerServices);
            $footerServices = Service::footer()->active()->take(4)->get();
            View::share('footerServices', $footerServices);
            $relatedServices = Service::active()->get();
            View::share('relatedServices', $relatedServices);
        }

        if (Schema::hasTable('clients')) {
            $clients = Client::active()->get();
            View::share('clients', $clients);
        }

        if (Schema::hasTable('parteners')) {
            $parteners = Partener::active()->get();
            View::share('parteners', $parteners);
        }

        if (Schema::hasTable('products')) {
            $headerProducts = Product::header()->active()->take(4)->get();
            View::share('headerProducts', $headerProducts);
            $footerProducts = Product::footer()->active()->take(4)->get();
            View::share('footerProducts', $footerProducts);
        }

        if (Schema::hasTable('projects')) {
            $headerProjects = Project::header()->active()->take(4)->get();
            View::share('headerProjects', $headerProjects);
            $footerProjects = Project::footer()->active()->take(4)->get();
            View::share('footerProjects', $footerProjects);
        }

        if (Schema::hasTable('sliders')) {
            $top_header = Slider::TopHeader()->active()->first();
            View::share('top_header', $top_header);
        }
        if (Schema::hasTable('phones')) {
            $phones = Phone::active()->get();
            View::share('phones', $phones);
        }
        if(Schema::hasTable('sections')){
            $sections = Section::all();
            View::share('sections', $sections);
        }

        if (Schema::hasTable('blogs')) {
            Blog::observe(BlogObserver::class);
            $blogs = Blog::active()->get();
            View::share('blogs', $blogs);
        }

        if (Schema::hasTable('products')) {
            Product::observe(ProductObserver::class);
        }

        if (Schema::hasTable('projects')) {
            Project::observe(ProjectObserver::class);
        }

        if (Schema::hasTable('services')) {
            Service::observe(ServiceObserver::class);
        }

        // Ensure analytics permissions exist
        $this->ensureAnalyticsPermissionsExist();
    }

    /**
     * Ensure analytics permissions exist
     */
    private function ensureAnalyticsPermissionsExist(): void
    {
        try {
            if (Schema::hasTable('permissions') && Schema::hasTable('roles')) {
                $permissions = ['analytics.view', 'analytics.manage'];
                
                foreach($permissions as $permission) {
                    Permission::firstOrCreate([
                        'name' => $permission, 
                        'guard_name' => 'admin'
                    ]);
                }
                
                $adminRole = Role::firstOrCreate([
                    'name' => 'admin', 
                    'guard_name' => 'admin'
                ]);
                
                $adminRole->givePermissionTo($permissions);
            }
        } catch (\Exception $e) {
            // Log error if needed
            // Log::error('Failed to create analytics permissions: ' . $e->getMessage());
        }
    }
}