<?php

namespace App\Models;

use App\Traits\HasLanguage;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Testing\Fluent\Concerns\Has;
use Mcamara\LaravelLocalization\Facades\LaravelLocalization;

class Product extends Model
{
    /** @use HasFactory<\Database\Factories\ProductFactory> */
    use HasFactory ,HasLanguage;
    protected $table = 'products' ;
    protected $fillable = [
        'category_id',
        'parent_id',
        'name_en',
        'name_ar',
        'short_desc_en',
        'short_desc_ar',
        'long_desc_en',
        'long_desc_ar',
        'image',
        'alt_image',
        'icon',
        'alt_icon',
        'status',
        'show_in_home',
        'show_in_header',
        'show_in_footer',
        'meta_title_ar',
        'meta_title_en',
        'meta_desc_ar',
        'meta_desc_en',
        'index',
        'slug_en',
        'slug_ar',
        'order',
    ];

    public function getRouteKeyName()
    {
        if (!request()->is('*dashboard*')) {
            return 'slug_' . app()->getLocale(); // for frontend
        }
    
        return 'id'; // for dashboard/admin
    }

    public function parent()
    {
        return $this->belongsTo(Product::class, 'parent_id');
    }
    public function children()
    {
        return $this->hasMany(Product::class, 'parent_id');
    }
    public function category()
    {
        return $this->belongsTo(Category::class);
    }
   
    public function getImagePathAttribute()
    {
        return $this->image ? asset('storage/products/' . $this->image) : asset('assets/dashboard/images/noimage.png');
    }
    public function getIconPathAttribute()
    {
        return $this->icon ? asset('storage/products/' . $this->icon) : asset('assets/dashboard/images/noimage.png');
    }

    /**
     * Generate SEO-friendly alt text for main image
     */
    public function getImageAltAttribute()
    {
        if ($this->alt_image) {
            return $this->alt_image;
        }
        
        return $this->name . ' - ' . __('website.product') . ' | ' . config('settings.site_name');
    }

    /**
     * Generate SEO-friendly alt text for icon
     */
    public function getIconAltAttribute()
    {
        if ($this->alt_icon) {
            return $this->alt_icon;
        }
        
        return $this->name . ' - ' . __('website.product_icon') . ' | ' . config('settings.site_name');
    }

    public function getNameAttribute()
    {
        return $this->{'name_' . $this->lang};
    }
    public function getShortDescAttribute()
    {
        return $this->{'short_desc_' . $this->lang};
    }
    public function getLongDescAttribute()
    {
        return $this->{'long_desc_' . $this->lang};
    }

    public function scopeActive($query)
    {
        return $query->where('status', true);
    }

    public function scopeHome($query)
    {
        return $query->where('show_in_home', true);
    }

    public function scopeHeader($query)
    {
        return $query->where('show_in_header', true);
    }
    public function scopeFooter($query)
    {
        return $query->where('show_in_footer', true);
    }

        public function getLinkAttribute()
        {
            return  LaravelLocalization::localizeUrl('products/'.$this->{'slug_' . $this->lang});
        }
   
    public function productImages()
    {
        return $this->hasMany(ProductImage::class, 'product_id')->orderBy('order');
    }
    public function images()
    {
        return $this->productImages();
    }
    public function getClientsAttribute()
    {
        return $this->{'clients_' . $this->lang};
    }
    public function getlocationAttribute()
    {
        return $this->{'location_' . $this->lang};
    }
    public function getcategoryNameAttribute()
    {
        return $this->{'category_' . $this->lang};
    }
    public function getserviceAttribute()
    {
        return $this->{'service_' . $this->lang};
    }

}