<?php

namespace App\Services\Dashboard;

use App\Helper\Media;
use App\Models\Product;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;

class ProductService
{

    public function store($request)
    {
        $data = $request->validated();

        DB::beginTransaction();
        try {
          
            $data['slug_ar'] = Str::slug($data['name_ar']);
            $data['slug_en'] =  Str::slug($data['name_en']);

            if ($request->hasFile('image')) {
                $data['image'] = Media::uploadAndAttachImageStorage($request->file('image'), 'products');
            }
            if ($request->hasFile('icon')) {
                $data['icon'] = Media::uploadAndAttachImageStorage($request->file('icon'), 'products');
            }
            $product = Product::create($data);

            DB::commit();
            return $product;
        } catch (\Exception $e) {
            DB::rollback();
            throw $e;
        }
    }

    public function update($request, $data, $product)
    {
        DB::beginTransaction();
        try {
            $data['status'] = $data['status'] ?? 0;
            $data['show_in_home'] = $data['show_in_home'] ?? 0;
            $data['show_in_header'] = $data['show_in_header'] ?? 0;
            $data['show_in_footer'] = $data['show_in_footer'] ?? 0;
            $data['index'] = $data['index'] ?? 0;
            $data['slug_ar'] = Str::slug($data['slug_ar']);
            $data['slug_en'] = Str::slug($data['slug_en']);

            if ($request->hasFile('image')) {
                if ($product->image) {
                    Media::removeFile('products', $product->image);
                }
                $data['image'] = Media::uploadAndAttachImageStorage($request->file('image'), 'products');
            }

            if ($request->hasFile('icon')) {
                if ($product->icon) {
                    Media::removeFile('products', $product->icon);
                }
                $data['icon'] = Media::uploadAndAttachImageStorage($request->file('icon'), 'products');
            }

            $product->update($data);

            DB::commit();
            return $product;
        } catch (\Exception $e) {
            DB::rollback();
            throw $e;
        }
    }

    public function delete($selectedIds) 
    {
        Log::info('Starting delete in ProductService', [
            'selectedIds' => $selectedIds
        ]);

        $products = Product::whereIn('id', $selectedIds)->get();
        
        Log::info('Found products to delete', [
            'count' => $products->count(),
            'products' => $products->pluck('id')->toArray()
        ]);
    
        DB::beginTransaction();
        try {
            foreach ($products as $product) {
                Log::info('Processing product', [
                    'product_id' => $product->id,
                    'has_image' => !empty($product->image),
                    'has_icon' => !empty($product->icon)
                ]);

                // Delete associated product images
                $this->deleteProductImages($product);

                // Delete associated image if it exists
                if ($product->image) {
                    try {
                        Media::removeFile('products', $product->image);
                        Log::info('Deleted product image', [
                            'product_id' => $product->id,
                            'image' => $product->image
                        ]);
                    } catch (\Exception $e) {
                        Log::warning('Failed to delete product image', [
                            'product_id' => $product->id,
                            'image' => $product->image,
                            'error' => $e->getMessage()
                        ]);
                    }
                }
    
                // Delete associated Icon if it exists
                if ($product->icon) {
                    try {
                        Media::removeFile('products', $product->icon);
                        Log::info('Deleted product icon', [
                            'product_id' => $product->id,
                            'icon' => $product->icon
                        ]);
                    } catch (\Exception $e) {
                        Log::warning('Failed to delete product icon', [
                            'product_id' => $product->id,
                            'icon' => $product->icon,
                            'error' => $e->getMessage()
                        ]);
                    }
                }
    
                // Delete the product model (this triggers the observer)
                try {
                    $product->delete();
                    Log::info('Deleted product', [
                        'product_id' => $product->id
                    ]);
                } catch (\Exception $e) {
                    Log::error('Failed to delete product', [
                        'product_id' => $product->id,
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                    throw $e;
                }
            }
    
            DB::commit();
            Log::info('Delete operation completed successfully');
            return true;
    
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Delete operation failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    /**
     * Delete all images associated with a product
     */
    private function deleteProductImages(Product $product): void
    {
        try {
            $productImages = $product->productImages;
            
            foreach ($productImages as $image) {
                // Get the raw filename from the image
                $filename = $image->getImageFilenameAttribute();
                
                if ($filename) {
                    Media::removeFile('products', $filename);
                    Log::info('Deleted product image file', [
                        'product_id' => $product->id,
                        'image_id' => $image->id,
                        'filename' => $filename
                    ]);
                }
                
                $image->delete();
            }
            
            Log::info('Deleted all product images', [
                'product_id' => $product->id,
                'count' => $productImages->count()
            ]);
            
        } catch (\Exception $e) {
            Log::warning('Failed to delete some product images', [
                'product_id' => $product->id,
                'error' => $e->getMessage()
            ]);
        }
    }
    
}
