<?php

namespace App\Models;

use App\Traits\HasLanguage;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

use Illuminate\Database\Eloquent\Builder;
use Mcamara\LaravelLocalization\Facades\LaravelLocalization;

class Service extends Model
{
    /** @use HasFactory<\Database\Factories\ServiceFactory> */
    use HasFactory , HasLanguage;

    protected $table = 'services';
    protected $fillable = ['name_ar','name_en','parent_id','short_desc_ar','short_desc_en','long_desc_ar','long_desc_en','image','alt_image','icon','alt_icon','status','show_in_home','show_in_header','show_in_footer','slug_ar','slug_en','meta_title_ar','meta_title_en','meta_desc_ar','meta_desc_en','index','order'];

    public function getRouteKeyName()
    {
        if (!request()->is('*dashboard*')) {
            return 'slug_' . app()->getLocale(); // for frontend
        }
    
        return 'id'; // for dashboard/admin
    }

    public function parent()
    {
        return $this->belongsTo(Service::class,'parent_id');
    }

    public function serviceImages()
    {
        return $this->hasMany(ServiceImage::class, 'service_id')->orderBy('order');
    }

    public function images()
    {
        return $this->serviceImages();
    }

    public function getNameAttribute()
    {
        return $this->{'name_'.$this->lang};
    }

    public function getShortDescAttribute()
    {
        return $this->{'short_desc_'.$this->lang};
    }

    public function getLongDescAttribute()
    {
        return $this->{'long_desc_'.$this->lang};
    }

    public function getImagePathAttribute()
    {
        return $this->image? asset('storage/services/' .$this->image) : asset('assets/dashboard/images/noimage.png');
    }

    public function getIconPathAttribute()
    {
        return $this->icon ? asset('storage/services/' . $this->icon) : asset('assets/dashboard/images/noIcon.png');
    }

    /**
     * Generate SEO-friendly alt text for main image
     */
    public function getImageAltAttribute()
    {
        if ($this->alt_image) {
            return $this->alt_image;
        }
        
        return $this->name . ' - ' . __('website.service') . ' | ' . config('settings.site_name');
    }

    /**
     * Generate SEO-friendly alt text for icon
     */
    public function getIconAltAttribute()
    {
        if ($this->alt_icon) {
            return $this->alt_icon;
        }
        
        return $this->name . ' - ' . __('website.service_icon') . ' | ' . config('settings.site_name');
    }

    public function scopeActive(Builder $query): void
    {
        $query->where('status', 1);
    }

    public function scopeHome(Builder $query): void
    {
        $query->where('show_in_home', 1);
    }

    public function scopeFooter(Builder $query): void
    {
        $query->where('show_in_footer', 1);
    }
    public function scopeHeader(Builder $query): void
    {
        $query->where('show_in_header', 1);
    }



    public function getLinkAttribute()
    {
        return  LaravelLocalization::LocalizeUrl('services/'.$this->{'slug_' . $this->lang});
    }
}
