<?php

namespace App\Models;

use App\Traits\HasLanguage;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Mcamara\LaravelLocalization\Facades\LaravelLocalization;

class Blog extends Model
{
    /** @use HasFactory<\Database\Factories\BlogFactory> */
    use HasFactory,HasLanguage;
    protected $table = 'blogs';
    protected $fillable = [
        'category_id',
        'author_id',
        'name_en',
        'name_ar',
        'short_desc_en',
        'short_desc_ar',
        'long_desc_en',
        'long_desc_ar',
        'image',
        'alt_image',
        'icon',
        'alt_icon',
        'meta_title_en',
        'meta_title_ar',
        'meta_desc_en',
        'meta_desc_ar',
        'slug_en',
        'slug_ar',
        'status',
        'show_in_home',
        'show_in_header',
        'show_in_footer',
        'index',
        'date',
        'order',
        
    ];
    protected $casts = [
        'date' => 'datetime',
    ];
    public function getRouteKeyName()
    {
        if (!request()->is('*dashboard*')) {
            return 'slug_' . app()->getLocale(); // for frontend
        }
    
        return 'id'; // for dashboard/admin
    }

    public function getDateAttribute()
    {
        $original = $this->getAttributes()['date'] ?? null;
    
        return $original ? \Carbon\Carbon::parse($original)->format('Y-m-d') : '';
    }
    public function category()
    {
        return $this->belongsTo(BlogCategory::class, 'category_id');
    }

    public function author()
    {
        return $this->belongsTo(Author::class, 'author_id');
    }
    
    public function getImagePathAttribute()
    {
        return $this->image ? asset('storage/blogs/' . $this->image) : asset('assets/dashboard/images/noimage.png');
    }

    public function getIconPathAttribute()
    {
        return $this->icon ? asset('storage/blogs/' . $this->icon) : asset('assets/dashboard/images/noimage.png');
    }

    /**
     * Generate SEO-friendly alt text for main image
     */
    public function getImageAltAttribute()
    {
        if ($this->alt_image) {
            return $this->alt_image;
        }
        
        return $this->name . ' - ' . __('website.blog_article') . ' | ' . config('settings.site_name');
    }

    /**
     * Generate SEO-friendly alt text for icon
     */
    public function getIconAltAttribute()
    {
        if ($this->alt_icon) {
            return $this->alt_icon;
        }
        
        return $this->name . ' - ' . __('website.blog_thumbnail') . ' | ' . config('settings.site_name');
    }

    public function getShortDescAttribute()
    {
        return $this->{'short_desc_'.$this->lang};

    }
    public function getLongDescAttribute()
    {
        return $this->{'long_desc_'.$this->lang};

    }
    public function getNameAttribute()
    {
        return $this->{'name_'.$this->lang};

    }
    public function getSlugAttribute()
    {
        return  LaravelLocalization::localizeUrl($this->{'slug_' . $this->lang});
    }


    public function scopeActive($query)
    {
        return $query->where('status', true);
    }
    public function scopeHome($query)
    {
        return $query->where('show_in_home', true);
    }
    public function scopeHeader($query)
    {
        return $query->where('show_in_header', true);
    }
    public function scopeFooter($query)
    {
        return $query->where('show_in_footer', true);
    }
    public function getLinkAttribute()
    {
        return  LaravelLocalization::localizeUrl('blogs/'.$this->{'slug_' . $this->lang});
    }
}
