<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Carbon;
use DOMDocument;
use DOMXPath;

class SeoTestingController extends Controller
{
    private string $baseUrl;

    public function __construct()
    {
        $this->baseUrl = rtrim(config('app.url', 'http://mepicom.test'), '/');
        
        // For local development, prefer HTTP over HTTPS to avoid SSL issues
        if (app()->environment('local') && strpos($this->baseUrl, 'https://') === 0) {
            $this->baseUrl = str_replace('https://', 'http://', $this->baseUrl);
        }
    }

    /**
     * Display SEO testing dashboard
     */
    public function index()
    {
        // Get recent test results if they exist
        $recentResults = $this->getRecentTestResults();
        
        return view('Dashboard.seo-testing.index', compact('recentResults'));
    }

    /**
     * Run comprehensive SEO test
     */
    public function runComprehensiveTest(Request $request)
    {
        try {
            $pages = [
                'home' => '/',
                'about' => '/about-us',
                'contact' => '/contact-us',
                'blogs' => '/blogs',
                'services' => '/services',
                'products' => '/products'
            ];

            $results = [];
            $totalScore = 0;
            $maxScore = 0;

            foreach ($pages as $name => $path) {
                $pageResult = $this->testPage($name, $path);
                $results[$name] = $pageResult;
                
                $totalScore += $pageResult['score'];
                $maxScore += $pageResult['max_score'];
            }

            // Test technical elements
            $technicalResults = $this->testTechnicalSEO();
            $results['technical'] = $technicalResults;
            $totalScore += $technicalResults['score'];
            $maxScore += $technicalResults['max_score'];

            $overallScore = $maxScore > 0 ? round(($totalScore / $maxScore) * 100, 1) : 0;

            $finalResults = [
                'overall_score' => $overallScore,
                'total_score' => $totalScore,
                'max_score' => $maxScore,
                'pages' => $results,
                'timestamp' => Carbon::now(),
                'recommendations' => $this->generateRecommendations($results)
            ];

            // Cache results for 1 hour
            cache(['seo_test_results' => $finalResults], 3600);

            return response()->json([
                'success' => true,
                'results' => $finalResults
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to run comprehensive test: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Run quick SEO test
     */
    public function runQuickTest(Request $request)
    {
        try {
            $url = $request->get('url', $this->baseUrl);
            $result = $this->testPage('quick_test', parse_url($url, PHP_URL_PATH) ?: '/');

            return response()->json([
                'success' => true,
                'result' => $result
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to run quick test: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Test sitemap functionality
     */
    public function testSitemap()
    {
        try {
            $sitemapUrls = [
                $this->baseUrl . '/sitemap.xml'
            ];

            $result = [
                'found' => false,
                'url' => null,
                'sitemap_count' => 0,
                'url_count' => 0,
                'last_modified' => null,
                'errors' => []
            ];

            foreach ($sitemapUrls as $url) {
                $content = $this->fetchContent($url);
                
                if ($content && (strpos($content, '<?xml') === 0 || strpos($content, '<sitemapindex') !== false)) {
                    $result['found'] = true;
                    $result['url'] = $url;
                    
                    // Parse sitemap
                    if (strpos($content, '<sitemapindex') !== false) {
                        preg_match_all('/<sitemap>.*?<loc>(.*?)<\/loc>/s', $content, $matches);
                        $result['sitemap_count'] = count($matches[1]);
                    } else {
                        preg_match_all('/<loc>(.*?)<\/loc>/', $content, $matches);
                        $result['url_count'] = count($matches[1]);
                    }
                    
                    // Get last modified
                    preg_match('/<lastmod>(.*?)<\/lastmod>/', $content, $lastModMatches);
                    if (!empty($lastModMatches[1])) {
                        $result['last_modified'] = $lastModMatches[1];
                    }
                    
                    break;
                }
            }

            return response()->json([
                'success' => true,
                'sitemap' => $result
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to test sitemap: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get SEO recommendations
     */
    public function getRecommendations()
    {
        try {
            // Get recent test results to generate specific recommendations
            $recentResults = $this->getRecentTestResults();
            
            if ($recentResults && isset($recentResults['pages'])) {
                $recommendations = $this->generateRecommendations($recentResults['pages']);
            } else {
                // Fallback to general recommendations
                $recommendations = [
                    'high_priority' => [
                        'Fix missing meta descriptions on key pages',
                        'Add structured data markup for better rich snippets',
                        'Optimize page load speeds (target under 3 seconds)',
                        'Ensure all images have descriptive alt tags'
                    ],
                    'medium_priority' => [
                        'Improve internal linking structure',
                        'Add Open Graph tags for social media sharing',
                        'Optimize title tag lengths (30-60 characters)',
                        'Create XML sitemap if missing'
                    ],
                    'low_priority' => [
                        'Add Twitter Card meta tags',
                        'Implement breadcrumb navigation',
                        'Add FAQ schema markup where applicable',
                        'Monitor Core Web Vitals metrics'
                    ]
                ];
            }

            return response()->json([
                'success' => true,
                'recommendations' => $recommendations
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to generate recommendations: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Test individual page SEO
     */
    private function testPage(string $pageName, string $path): array
    {
        $url = $this->baseUrl . $path;
        $html = $this->fetchContent($url);

        if (!$html) {
            return [
                'name' => $pageName,
                'url' => $url,
                'accessible' => false,
                'score' => 0,
                'max_score' => 1,
                'percentage' => 0,
                'tests' => [
                    ['name' => 'Page Loading', 'passed' => false, 'details' => 'Failed to load page']
                ]
            ];
        }

        $tests = [];
        $score = 0;
        $maxScore = 0;

        // Test meta tags
        $metaTests = $this->testMetaTags($html);
        $tests = array_merge($tests, $metaTests['tests']);
        $score += $metaTests['score'];
        $maxScore += $metaTests['max_score'];

        // Test page structure
        $structureTests = $this->testPageStructure($html);
        $tests = array_merge($tests, $structureTests['tests']);
        $score += $structureTests['score'];
        $maxScore += $structureTests['max_score'];

        // Test social media tags
        $socialTests = $this->testSocialMediaTags($html);
        $tests = array_merge($tests, $socialTests['tests']);
        $score += $socialTests['score'];
        $maxScore += $socialTests['max_score'];

        return [
            'name' => $pageName,
            'url' => $url,
            'accessible' => true,
            'score' => $score,
            'max_score' => $maxScore,
            'percentage' => $maxScore > 0 ? round(($score / $maxScore) * 100, 1) : 0,
            'tests' => $tests
        ];
    }

    /**
     * Test meta tags
     */
    private function testMetaTags(string $html): array
    {
        $dom = new DOMDocument();
        @$dom->loadHTML($html);
        $xpath = new DOMXPath($dom);

        $tests = [];
        $score = 0;
        $maxScore = 6;

        // Title tag
        $title = $xpath->query('//title')->item(0);
        if ($title && strlen($title->textContent) > 0) {
            $titleLength = strlen($title->textContent);
            if ($titleLength >= 30 && $titleLength <= 60) {
                $tests[] = ['name' => 'Title Tag', 'passed' => true, 'details' => "Good length: {$titleLength} chars"];
                $score++;
            } else {
                $tests[] = ['name' => 'Title Tag', 'passed' => false, 'details' => "Poor length: {$titleLength} chars"];
            }
        } else {
            $tests[] = ['name' => 'Title Tag', 'passed' => false, 'details' => 'Missing title tag'];
        }

        // Meta description
        $description = $xpath->query('//meta[@name="description"]/@content')->item(0);
        if ($description && strlen($description->value) > 0) {
            $descLength = strlen($description->value);
            if ($descLength >= 120 && $descLength <= 160) {
                $tests[] = ['name' => 'Meta Description', 'passed' => true, 'details' => "Good length: {$descLength} chars"];
                $score++;
            } else {
                $tests[] = ['name' => 'Meta Description', 'passed' => false, 'details' => "Poor length: {$descLength} chars"];
            }
        } else {
            $tests[] = ['name' => 'Meta Description', 'passed' => false, 'details' => 'Missing meta description'];
        }

        // Robots meta
        $robots = $xpath->query('//meta[@name="robots"]/@content')->item(0);
        if ($robots) {
            $tests[] = ['name' => 'Robots Meta', 'passed' => true, 'details' => "Content: {$robots->value}"];
            $score++;
        } else {
            $tests[] = ['name' => 'Robots Meta', 'passed' => false, 'details' => 'Missing robots meta'];
        }

        // Canonical URL
        $canonical = $xpath->query('//link[@rel="canonical"]/@href')->item(0);
        if ($canonical) {
            $tests[] = ['name' => 'Canonical URL', 'passed' => true, 'details' => 'Canonical URL present'];
            $score++;
        } else {
            $tests[] = ['name' => 'Canonical URL', 'passed' => false, 'details' => 'Missing canonical URL'];
        }

        // Viewport meta
        $viewport = $xpath->query('//meta[@name="viewport"]/@content')->item(0);
        if ($viewport) {
            $tests[] = ['name' => 'Viewport Meta', 'passed' => true, 'details' => 'Mobile viewport configured'];
            $score++;
        } else {
            $tests[] = ['name' => 'Viewport Meta', 'passed' => false, 'details' => 'Missing viewport meta'];
        }

        // Schema markup
        $hasSchema = strpos($html, 'application/ld+json') !== false;
        if ($hasSchema) {
            $tests[] = ['name' => 'Schema Markup', 'passed' => true, 'details' => 'Structured data found'];
            $score++;
        } else {
            $tests[] = ['name' => 'Schema Markup', 'passed' => false, 'details' => 'No structured data'];
        }

        return ['tests' => $tests, 'score' => $score, 'max_score' => $maxScore];
    }

    /**
     * Test page structure
     */
    private function testPageStructure(string $html): array
    {
        $dom = new DOMDocument();
        @$dom->loadHTML($html);
        $xpath = new DOMXPath($dom);

        $tests = [];
        $score = 0;
        $maxScore = 3;

        // H1 tag
        $h1Tags = $xpath->query('//h1');
        if ($h1Tags->length === 1) {
            $tests[] = ['name' => 'H1 Tag', 'passed' => true, 'details' => 'Single H1 found'];
            $score++;
        } else {
            $tests[] = ['name' => 'H1 Tag', 'passed' => false, 'details' => "Found {$h1Tags->length} H1 tags"];
        }

        // Heading hierarchy
        $headings = $xpath->query('//h1 | //h2 | //h3 | //h4 | //h5 | //h6');
        if ($headings->length > 0) {
            $tests[] = ['name' => 'Heading Structure', 'passed' => true, 'details' => "{$headings->length} headings found"];
            $score++;
        } else {
            $tests[] = ['name' => 'Heading Structure', 'passed' => false, 'details' => 'No headings found'];
        }

        // Alt tags
        $images = $xpath->query('//img');
        $imagesWithAlt = $xpath->query('//img[@alt]');
        if ($images->length > 0) {
            $altPercentage = round(($imagesWithAlt->length / $images->length) * 100);
            if ($altPercentage >= 90) {
                $tests[] = ['name' => 'Image Alt Tags', 'passed' => true, 'details' => "{$altPercentage}% images have alt tags"];
                $score++;
            } else {
                $tests[] = ['name' => 'Image Alt Tags', 'passed' => false, 'details' => "Only {$altPercentage}% images have alt tags"];
            }
        } else {
            $tests[] = ['name' => 'Image Alt Tags', 'passed' => true, 'details' => 'No images found'];
            $score++;
        }

        return ['tests' => $tests, 'score' => $score, 'max_score' => $maxScore];
    }

    /**
     * Test social media tags
     */
    private function testSocialMediaTags(string $html): array
    {
        $dom = new DOMDocument();
        @$dom->loadHTML($html);
        $xpath = new DOMXPath($dom);

        $tests = [];
        $score = 0;
        $maxScore = 2;

        // Open Graph tags
        $ogTags = ['og:title', 'og:description', 'og:image', 'og:url'];
        $ogFound = 0;
        foreach ($ogTags as $tag) {
            $element = $xpath->query("//meta[@property='$tag']/@content")->item(0);
            if ($element && strlen($element->value) > 0) {
                $ogFound++;
            }
        }

        if ($ogFound >= 3) {
            $tests[] = ['name' => 'Open Graph Tags', 'passed' => true, 'details' => "{$ogFound}/4 OG tags found"];
            $score++;
        } else {
            $tests[] = ['name' => 'Open Graph Tags', 'passed' => false, 'details' => "Only {$ogFound}/4 OG tags found"];
        }

        // Twitter Cards
        $twitterTags = ['twitter:card', 'twitter:title', 'twitter:description', 'twitter:image'];
        $twitterFound = 0;
        foreach ($twitterTags as $tag) {
            $element = $xpath->query("//meta[@name='$tag']/@content")->item(0);
            if ($element && strlen($element->value) > 0) {
                $twitterFound++;
            }
        }

        if ($twitterFound >= 3) {
            $tests[] = ['name' => 'Twitter Cards', 'passed' => true, 'details' => "{$twitterFound}/4 Twitter tags found"];
            $score++;
        } else {
            $tests[] = ['name' => 'Twitter Cards', 'passed' => false, 'details' => "Only {$twitterFound}/4 Twitter tags found"];
        }

        return ['tests' => $tests, 'score' => $score, 'max_score' => $maxScore];
    }

    /**
     * Test technical SEO
     */
    private function testTechnicalSEO(): array
    {
        $tests = [];
        $score = 0;
        $maxScore = 3;

        // Test sitemap
        $sitemapUrls = [
            $this->baseUrl . '/sitemap.xml'
        ];

        $sitemapFound = false;
        foreach ($sitemapUrls as $url) {
            $content = $this->fetchContent($url);
            if ($content && (strpos($content, '<?xml') === 0 || strpos($content, '<sitemapindex') !== false)) {
                $tests[] = ['name' => 'XML Sitemap', 'passed' => true, 'details' => 'Sitemap accessible'];
                $score++;
                $sitemapFound = true;
                break;
            }
        }

        if (!$sitemapFound) {
            $tests[] = ['name' => 'XML Sitemap', 'passed' => false, 'details' => 'Sitemap not accessible'];
        }

        // Test robots.txt
        $robotsUrls = [
            $this->baseUrl . '/robots.txt'
        ];

        $robotsFound = false;
        foreach ($robotsUrls as $url) {
            $content = $this->fetchContent($url);
            if ($content && strpos($content, 'User-agent:') !== false) {
                $tests[] = ['name' => 'Robots.txt', 'passed' => true, 'details' => 'Robots.txt accessible'];
                $score++;
                $robotsFound = true;
                break;
            }
        }

        if (!$robotsFound) {
            $tests[] = ['name' => 'Robots.txt', 'passed' => false, 'details' => 'Robots.txt not found'];
        }

        // Test page speed
        $startTime = microtime(true);
        $this->fetchContent($this->baseUrl);
        $loadTime = microtime(true) - $startTime;

        if ($loadTime <= 3.0) {
            $tests[] = ['name' => 'Page Speed', 'passed' => true, 'details' => number_format($loadTime, 2) . ' seconds'];
            $score++;
        } else {
            $tests[] = ['name' => 'Page Speed', 'passed' => false, 'details' => number_format($loadTime, 2) . ' seconds (slow)'];
        }

        return [
            'name' => 'Technical SEO',
            'score' => $score,
            'max_score' => $maxScore,
            'percentage' => $maxScore > 0 ? round(($score / $maxScore) * 100, 1) : 0,
            'tests' => $tests
        ];
    }

    /**
     * Fetch content from URL
     */
    private function fetchContent(string $url): ?string
    {
        // Try HTTPS first, then HTTP if HTTPS fails (common in local development)
        $urls = [$url];
        if (strpos($url, 'https://') === 0) {
            $urls[] = str_replace('https://', 'http://', $url);
        }
        
        foreach ($urls as $tryUrl) {
            try {
                // Use Laravel's HTTP client for better error handling
                $response = Http::withOptions([
                    'verify' => config('app.env') !== 'local', // Only disable SSL verification in local development
                    'timeout' => 30,
                ])->get($tryUrl);
                    
                if ($response->successful()) {
                    return $response->body();
                }
                
                Log::warning("SEO Testing - Non-successful response from: $tryUrl", [
                    'status' => $response->status(),
                    'body' => substr($response->body(), 0, 200)
                ]);
                
            } catch (\Exception $e) {
                // Log error for debugging
                Log::info("SEO Testing - Failed to fetch content from: $tryUrl", [
                    'error' => $e->getMessage()
                ]);
                // Continue to try next URL
            }
        }
        
        return null;
    }

    /**
     * Generate recommendations based on test results
     */
    private function generateRecommendations(array $results): array
    {
        $recommendations = [
            'high_priority' => [],
            'medium_priority' => [],
            'low_priority' => []
        ];

        foreach ($results as $pageName => $pageData) {
            if (!is_array($pageData) || !isset($pageData['tests'])) continue;

            foreach ($pageData['tests'] as $test) {
                if (!$test['passed']) {
                    switch ($test['name']) {
                        case 'Meta Description':
                            $recommendations['high_priority'][] = "Add meta description to {$pageName} page";
                            break;
                        case 'Title Tag':
                            $recommendations['high_priority'][] = "Optimize title tag on {$pageName} page";
                            break;
                        case 'Schema Markup':
                            $recommendations['medium_priority'][] = "Add structured data to {$pageName} page";
                            break;
                        case 'Open Graph Tags':
                            $recommendations['medium_priority'][] = "Add Open Graph tags to {$pageName} page";
                            break;
                        case 'Twitter Cards':
                            $recommendations['low_priority'][] = "Add Twitter Cards to {$pageName} page";
                            break;
                    }
                }
            }
        }

        // Remove duplicates
        foreach ($recommendations as $priority => $items) {
            $recommendations[$priority] = array_unique($items);
        }

        return $recommendations;
    }

    /**
     * Get recent test results from cache
     */
    private function getRecentTestResults(): ?array
    {
        return cache('seo_test_results');
    }
}