<?php

namespace App\Http\Controllers\Website;

use App\Http\Controllers\Controller;
use App\Http\Requests\Website\SaveCareerApplicationRequest;
use App\Http\Requests\Website\StoreContactUsRequest;
use App\Models\Benefit;
use App\Models\Blog;
use App\Models\Dashboard\AboutStruct;
use App\Models\Dashboard\AboutUs;
use App\Models\Dashboard\Domain;
use App\Models\Faq;
use App\Models\GalleryVideo;
use App\Models\JobPosition;
use App\Models\Partener;
use App\Models\Phone;
use App\Models\Product;
use App\Models\Project;
use App\Models\Service;
use App\Models\SiteAddress;
use App\Models\Statistic;
use App\Models\Team;
use App\Services\Dashboard\SaveApplicationService;
use App\Services\Website\ContactUsService;
use App\Services\Website\StoreContactUsService;
use Illuminate\Http\Request;
use App\Models\Category;
use App\Models\Section;
use App\Models\Album;

class WebsiteController extends Controller
{
    /**
     * Get social media links from configuration
     */
    private function getSocialMediaLinks()
    {
        return [
            'whatsapp' => config('settings.site_whatsapp') ? 'https://wa.me/' . ltrim(config('settings.site_whatsapp'), '+') : '#',
            'facebook' => config('settings.site_facebook') ?? '#',
            'twitter' => config('settings.site_twitter') ?? '#',
            'instagram' => config('settings.site_instagram') ?? '#',
            'youtube' => config('settings.site_youtube') ?? '#',
            'linkedin' => config('settings.site_linkedin') ?? '#',
            'tiktok' => config('settings.site_tiktok') ?? '#',
            'snapchat' => config('settings.site_snapchat') ?? '#',
            'pinterest' => config('settings.site_pinterest') ?? '#',
            'telegram' => config('settings.site_telegram') ?? '#',
        ];
    }
    public function about()
    {
        $data['about'] = AboutUs::first();
        $data['about_structs'] = AboutStruct::active()->get();
        $data['faqs'] = Faq::active()->general()->get();
        $data['statistics'] = Statistic::active()->take(4)->get();
        $data['teams'] = Team::active()->get();

        $data['socialMediaLinks'] = $this->getSocialMediaLinks();
        return view('Website.about', $data);
    }

    public function services()
    {
        $data['services'] = Service::active()->get();
        $data['sections'] = Section::all();
        $data['socialMediaLinks'] = $this->getSocialMediaLinks();

        return view('Website.services', $data);
    }




    public function serviceDetails($slug)
    {
        $locale = app()->getLocale();

        // Decode URL-encoded characters and fix mojibake
        $decodedSlug = urldecode($slug);

        // Fix Arabic mojibake encoding issues
        $fixedSlug = $this->fixArabicMojibake($decodedSlug);

        // Try to find service using JSON_EXTRACT for current locale
        $service = Service::whereRaw("JSON_EXTRACT(slug, '$.{$locale}') = ?", [$fixedSlug])->first();

        // If not found, try with original decoded slug
        if (!$service) {
            $service = Service::whereRaw("JSON_EXTRACT(slug, '$.{$locale}') = ?", [$decodedSlug])->first();
        }

        // If not found, try to find by checking all services manually (fallback for double JSON issue)
        if (!$service) {
            $services = Service::all();
            foreach ($services as $s) {
                $serviceSlug = $s->getTranslation('slug', $locale);
                if ($serviceSlug === $fixedSlug || $serviceSlug === $decodedSlug) {
                    $service = $s;
                    break;
                }
            }
        }

        // If still not found, try to find by checking all languages
        if (!$service) {
            $services = Service::all();
            foreach ($services as $s) {
                foreach (['en', 'ar', 'fr'] as $lang) {
                    $serviceSlug = $s->getTranslation('slug', $lang);
                    if ($serviceSlug === $fixedSlug || $serviceSlug === $decodedSlug) {
                        $service = $s;
                        break 2;
                    }
                }
            }
        }

        if (!$service) {
            abort(404);
        }

        // The middleware will handle slug redirections, so we just display the service

        // #call# placeholder will be replaced automatically by the model accessor
        $socialMediaLinks = $this->getSocialMediaLinks();
        return view('Website.service-details', compact('service', 'socialMediaLinks'));
    }

    /**
     * Fix Arabic mojibake encoding issues
     */
    private function fixArabicMojibake($text)
    {
        // Try safe encoding conversion
        $supportedEncodings = ['ISO-8859-6', 'UTF-8'];
        foreach ($supportedEncodings as $encoding) {
            if (mb_check_encoding($text, $encoding)) {
                $converted = @mb_convert_encoding($text, 'UTF-8', $encoding);
                if ($converted && $converted !== $text && mb_strlen($converted) > 0) {
                    $text = $converted;
                    break;
                }
            }
        }

        // Common mojibake pattern fixes
        $patterns = [
            'Ø§' => 'ا',
            'Ø¨' => 'ب',
            'Øª' => 'ت',
            'Ø«' => 'ث',
            'Ø¬' => 'ج',
            'Ø' => 'ح',
            'Ø®' => 'خ',
            'Ø¯' => 'د',
            'Ø°' => 'ذ',
            'Ø±' => 'ر',
            'Ø²' => 'ز',
            'Ø³' => 'س',
            'Ø´' => 'ش',
            'Øµ' => 'ص',
            'Ø¶' => 'ض',
            'Ø·' => 'ط',
            'Ø¸' => 'ظ',
            'Ø¹' => 'ع',
            'Øº' => 'غ',
            'Ø¡' => 'ء',
            'Ø¢' => 'آ',
            'Ø£' => 'أ',
            'Ø¥' => 'إ',
            'Ø¦' => 'ئ',
            'Ø®Ù' => 'خد',
            'Ø§Ø®Ø¯' => 'اخد',
            'Ù%C2%85' => 'ة'
        ];

        return str_replace(array_keys($patterns), array_values($patterns), $text);
    }

    public function categories()
    {
        $data['categories'] = Category::active()->get();
        $data['services'] = Service::active()->get();
        $data['socialMediaLinks'] = $this->getSocialMediaLinks();
        return view('Website.categories', $data);
    }

    public function categoryDetails($slug)
    {
        $locale = app()->getLocale();

        // Decode URL-encoded Arabic characters
        $decodedSlug = urldecode($slug);

        // Try to find category using JSON_EXTRACT for current locale
        $category = Category::whereRaw("JSON_EXTRACT(slug, '$.{$locale}') = ?", [$decodedSlug])->first();

        // If not found, try to find by checking all categories manually (fallback for double JSON issue)
        if (!$category) {
            $categories = Category::all();
            foreach ($categories as $c) {
                $categorySlug = $c->getTranslation('slug', $locale);
                if ($categorySlug === $decodedSlug) {
                    $category = $c;
                    break;
                }
            }
        }

        // If still not found, try to find by checking all languages
        if (!$category) {
            $categories = Category::all();
            foreach ($categories as $c) {
                foreach (['en', 'ar', 'fr'] as $lang) {
                    $categorySlug = $c->getTranslation('slug', $lang);
                    if ($categorySlug === $decodedSlug) {
                        $category = $c;
                        break 2;
                    }
                }
            }
        }

        if (!$category) {
            abort(404);
        }

        // #call# placeholder will be replaced automatically by the model accessor
        $socialMediaLinks = $this->getSocialMediaLinks();
        return view('Website.category-details', compact('category', 'socialMediaLinks'));
    }

    public function showContactUs()
    {
        $data['site_addresses'] = SiteAddress::active()->orderBy('order')->get();
        $data['phones'] = Phone::active()->orderBy('order')->get();
        $data['products'] = Product::active()->main()->get();

        // Get sections
        $sections = Section::all();

        $data['sections'] = $sections;
        $data['socialMediaLinks'] = $this->getSocialMediaLinks();
        return view('Website.contact-us', $data);
    }

    public function saveContactUs(StoreContactUsRequest $request)
    {
        try {

            $data = $request->validated();

            $response = (new ContactUsService)->store($data);

            if (!$response) {
                return redirect()->back()->with(['error' => __('website.failed_to_send_message')]);
            }

            return redirect()->back()->with(['success' => __('website.thanks_message')]);
        } catch (\Exception $e) {

            return redirect()->back()->with(['error' => __('website.something wrong pls try letter')]);
        }
    }

    public function suppliers()
    {
        $data['suppliers'] = Partener::active()->get();

        return view('Website.suppliers', $data);
    }
    public function products()
    {
        $data['products'] = Product::active()->main()->get();
        $data['socialMediaLinks'] = $this->getSocialMediaLinks();

        return view('Website.products', $data);
    }
    public function productDetails($slug)
    {
         $locale = app()->getLocale();
                 // Decode URL-encoded Arabic characters
        $decodedSlug = urldecode($slug);


        // Try to find product using JSON_EXTRACT
        $product = Product::whereRaw("JSON_EXTRACT(slug, '$.{$locale}') = ?", [$decodedSlug])->first();

        // If not found, try to find by checking all products manually (fallback for double JSON issue)
        if (!$product) {
            $products = Product::all();
            foreach ($products as $p) {
                $productSlug = $p->getTranslation('slug', $locale);
                if ($productSlug === $decodedSlug) {
                    $product = $p;
                    break;
                }
            }
        }

        if (!$product) {
            abort(404);
        }

        $suppliers = Partener::whereHas('products', function ($query) use($product){
            $query->where('parent_id' , $product->id   );
        })->active()->with('products')->get();
        $productsD = Product::where('parent_id', $product->id)->active()->get();
        // #call# placeholder will be replaced automatically by the model accessor
        $socialMediaLinks = $this->getSocialMediaLinks();
        return view('Website.product-details', compact('product', 'productsD','suppliers','socialMediaLinks'));
    }

    public function projects()
    {
        $data['projects'] = Project::active()->get();
        $data['socialMediaLinks'] = $this->getSocialMediaLinks();
        return view('Website.projects', $data);
    }
    public function projectDetails($slug)
    {
        $locale = app()->getLocale();

        // Decode URL-encoded Arabic characters
        $decodedSlug = urldecode($slug);

        // Try to find project using JSON_EXTRACT
        $project = Project::whereRaw("JSON_EXTRACT(slug, '$.{$locale}') = ?", [$decodedSlug])->first();

        // If not found, try to find by checking all projects manually (fallback for double JSON issue)
        if (!$project) {
            $projects = Project::all();
            foreach ($projects as $p) {
                $projectSlug = $p->getTranslation('slug', $locale);
                if ($projectSlug === $decodedSlug) {
                    $project = $p;
                    break;
                }
            }
        }

        if (!$project) {
            abort(404);
        }

        // #call# placeholder will be replaced automatically by the model accessor
        $socialMediaLinks = $this->getSocialMediaLinks();
        return view('Website.project-details', compact('project', 'socialMediaLinks'));
    }

    public function blogs()
    {
        $data['blogs'] = Blog::active()->orderBy('created_at', 'desc')->paginate(9);
        $data['sections'] = \App\Models\Section::all();
        $data['socialMediaLinks'] = $this->getSocialMediaLinks();
        return view('Website.blogs', $data);
    }
    public function blogDetails($slug)
    {
        $locale = app()->getLocale();

        // Decode URL-encoded Arabic characters
        $decodedSlug = urldecode($slug);

        // Try to find blog using JSON_EXTRACT for current locale
        $blog = Blog::whereRaw("JSON_EXTRACT(slug, '$.{$locale}') = ?", [$decodedSlug])->first();

        // If not found, try to find by checking all blogs manually (fallback for double JSON issue)
        if (!$blog) {
            $blogs = Blog::all();
            foreach ($blogs as $b) {
                $blogSlug = $b->getTranslation('slug', $locale);
                if ($blogSlug === $decodedSlug) {
                    $blog = $b;
                    break;
                }
            }
        }

        // If still not found, try to find by checking all languages
        if (!$blog) {
            $blogs = Blog::all();
            foreach ($blogs as $b) {
                foreach (['en', 'ar', 'fr'] as $lang) {
                    $blogSlug = $b->getTranslation('slug', $lang);
                    if ($blogSlug === $decodedSlug) {
                        $blog = $b;
                        break 2;
                    }
                }
            }
        }

        if (!$blog) {
            abort(404);
        }

        // #call# placeholder will be replaced automatically by the model accessor

        // Get related blogs (excluding current blog)
        $blogs = Blog::active()->where('id', '!=', $blog->id)->limit(5)->get();
        $socialMediaLinks = $this->getSocialMediaLinks();

        return view('Website.blog-details', compact('blog', 'blogs', 'socialMediaLinks'));
    }

    public function careers()
    {
        $data['gallery_videos'] = GalleryVideo::active()->get();
        $data['job_positions'] = JobPosition::active()->get();
        $data['socialMediaLinks'] = $this->getSocialMediaLinks();
        return view('Website.careers', $data);
    }

    public function saveApplication(SaveCareerApplicationRequest $request)
    {
        try {

            $response = (new SaveApplicationService)->saveApplication($request);

            if (!$response) {
                return redirect()->back()->with(['error' => __('website.failed_to_send_message')]);
            }

            return redirect()->back()->with(['success' => __('website.thanks_message')]);
        } catch (\Exception $e) {

            return redirect()->back()->with(['error' => __('website.something wrong pls try letter')]);
        }
    }

    public function galleryPhotos()
    {
        $data['gallery_photos'] = Album::active()->with('images')->get();
        $data['sections'] = \App\Models\Section::all();
        $data['socialMediaLinks'] = $this->getSocialMediaLinks();
        return view('Website.gallery-photos', $data);
    }

    public function galleryVideos()
    {
        $data['gallery_videos'] = GalleryVideo::active()->get();
        $data['sections'] = \App\Models\Section::all();
        $data['socialMediaLinks'] = $this->getSocialMediaLinks();
        return view('Website.gallery-videos', $data);
    }

    public function faqs()
    {
        $data['faqs'] = \App\Models\Faq::active()->orderBy('order')->get();
        $data['sections'] = \App\Models\Section::all();
        $data['socialMediaLinks'] = $this->getSocialMediaLinks();
        return view('Website.faqs', $data);
    }
}