<?php

namespace App\Http\Controllers;

use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class ManifestController extends Controller
{
    /**
     * Generate dynamic manifest.json
     */
    public function manifest(): JsonResponse
    {
        // Get current locale
        $currentLocale = app()->getLocale();

        // Get site settings from database (general settings)
        $settings = Setting::where('lang', 'all')->pluck('value', 'key')->toArray();

        // Get PWA settings for current language
        $pwaSettings = Setting::whereIn('key', ['site_short_name', 'site_description', 'theme_color', 'background_color', 'site_logo'])
            ->where('lang', $currentLocale)
            ->pluck('value', 'key')
            ->toArray();

        // Get site name and description from current locale settings
        $siteName = $pwaSettings['site_name'] ?? config("configrations.site_name") ?? ($currentLocale === 'ar' ? 'موقعي' : 'My Website');
        $siteDescription = $pwaSettings['site_description'] ?? config("configrations.site_description") ?? 'Professional web development, SEO, and digital marketing services in Egypt';
        $siteShortName = $pwaSettings['site_short_name'] ?? config("configrations.site_name") ?? 'Tulip';

        // Get theme colors
        $themeColor = $pwaSettings['theme_color'] ?? $settings['theme_color'] ?? '#007bff';
        $backgroundColor = $pwaSettings['background_color'] ?? $settings['background_color'] ?? '#ffffff';

        // Use PWA icons we created
        $logoUrl = asset('assets/website/images/icon-192x192.svg');

        // Get current locale
        $currentLocale = app()->getLocale();
        $isRTL = $currentLocale === 'ar';

        // Generate shortcuts based on available pages/menus
        $shortcuts = $this->generateShortcuts($currentLocale);

        // Generate icons array
        $icons = $this->generateIcons($logoUrl);

        // Get current base URL dynamically
        $baseUrl = url('/');
        $currentPath = request()->getPathInfo();

        // Extract subdirectory automatically from the current URL
        $subdirectory = '';
        $parsedUrl = parse_url($baseUrl);
        $path = $parsedUrl['path'] ?? '';

        // Remove trailing slash and get the subdirectory
        $path = rtrim($path, '/');
        if ($path && $path !== '/') {
            $subdirectory = $path;
        }

        $manifest = [
            'name' => $siteName,
            'short_name' => $siteShortName,
            'description' => $siteDescription,
            'start_url' => $subdirectory . '/' . $currentLocale,
            'display' => 'standalone',
            'background_color' => $backgroundColor,
            'theme_color' => $themeColor,
            'orientation' => 'portrait-primary',
            'icons' => $icons,
            'categories' => [
                'business',
                'productivity',
                'utilities'
            ],
            'lang' => $currentLocale,
            'dir' => $isRTL ? 'rtl' : 'ltr',
            'scope' => $subdirectory . '/',
            'prefer_related_applications' => false,
            'related_applications' => [],
            'screenshots' => [],
            'shortcuts' => $shortcuts
        ];

        return response()->json($manifest)
            ->header('Content-Type', 'application/manifest+json')
            ->header('Cache-Control', 'no-cache, no-store, must-revalidate') // Force refresh
            ->header('Pragma', 'no-cache')
            ->header('Expires', '0');
    }

    /**
     * Generate shortcuts based on available menus/pages
     */
    private function generateShortcuts(string $locale): array
    {
        $shortcuts = [];

        // Get current base URL dynamically
        $baseUrl = url('/');
        $parsedUrl = parse_url($baseUrl);
        $path = $parsedUrl['path'] ?? '';

        // Remove trailing slash and get the subdirectory
        $path = rtrim($path, '/');
        $subdirectory = ($path && $path !== '/') ? $path : '';

        // Home shortcut
        $shortcuts[] = [
            'name' => $locale === 'ar' ? 'الصفحة الرئيسية' : 'Home',
            'short_name' => $locale === 'ar' ? 'الرئيسية' : 'Home',
            'description' => $locale === 'ar' ? 'الصفحة الرئيسية للموقع' : 'Website Homepage',
            'url' => $subdirectory . '/' . $locale,
            'icons' => [
                [
                    'src' => asset('favicon.ico'),
                    'sizes' => '16x16 32x32 48x48'
                ]
            ]
        ];

        // Services shortcut
        $shortcuts[] = [
            'name' => $locale === 'ar' ? 'الخدمات' : 'Services',
            'short_name' => $locale === 'ar' ? 'الخدمات' : 'Services',
            'description' => $locale === 'ar' ? 'خدماتنا المتاحة' : 'Our Services',
            'url' => $subdirectory . '/' . $locale . '/services',
            'icons' => [
                [
                    'src' => asset('favicon.ico'),
                    'sizes' => '16x16 32x32 48x48'
                ]
            ]
        ];

        // Products shortcut
        $shortcuts[] = [
            'name' => $locale === 'ar' ? 'المنتجات' : 'Products',
            'short_name' => $locale === 'ar' ? 'المنتجات' : 'Products',
            'description' => $locale === 'ar' ? 'منتجاتنا المتاحة' : 'Our Products',
            'url' => $subdirectory . '/' . $locale . '/products',
            'icons' => [
                [
                    'src' => asset('favicon.ico'),
                    'sizes' => '16x16 32x32 48x48'
                ]
            ]
        ];

        // Contact shortcut
        $shortcuts[] = [
            'name' => $locale === 'ar' ? 'اتصل بنا' : 'Contact Us',
            'short_name' => $locale === 'ar' ? 'اتصل بنا' : 'Contact',
            'description' => $locale === 'ar' ? 'تواصل معنا' : 'Get in Touch',
            'url' => $subdirectory . '/' . $locale . '/contact',
            'icons' => [
                [
                    'src' => asset('favicon.ico'),
                    'sizes' => '16x16 32x32 48x48'
                ]
            ]
        ];

        return $shortcuts;
    }

    /**
     * Generate icons array based on available logo
     */
    private function generateIcons(string $logoUrl): array
    {
        $icons = [];

        // Add favicon
        $icons[] = [
            'src' => asset('favicon.ico'),
            'sizes' => '16x16 32x32 48x48',
            'type' => 'image/x-icon'
        ];

        // Add PWA icons with different sizes
        $pwaIcons = [
            ['size' => '152x152', 'file' => 'icon-152x152.svg'],
            ['size' => '192x192', 'file' => 'icon-192x192.svg'],
            ['size' => '512x512', 'file' => 'icon-512x512.svg']
        ];

        foreach ($pwaIcons as $icon) {
            $iconUrl = asset('assets/website/images/' . $icon['file']);

            // Add both any and maskable purposes
            $icons[] = [
                'src' => $iconUrl,
                'sizes' => $icon['size'],
                'type' => 'image/svg+xml',
                'purpose' => 'any'
            ];

            $icons[] = [
                'src' => $iconUrl,
                'sizes' => $icon['size'],
                'type' => 'image/svg+xml',
                'purpose' => 'maskable'
            ];
        }

        return $icons;
    }
}
