<?php

namespace App\Http\Middleware;

use App\Models\Redirect;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;

class RedirectsMiddleware
{
    public function handle(Request $request, Closure $next)
    {
        $path = '/' . ltrim($request->getPathInfo(), '/');
        $fullUrl = $request->fullUrl();
        $queryString = $request->getQueryString();

        // Build path with query string if exists
        $pathWithQuery = $path;
        if ($queryString) {
            $pathWithQuery = $path . '?' . $queryString;
        }

        // Also check decoded versions for URLs with encoded characters
        $decodedPath = urldecode($path);
        $decodedFullUrl = urldecode($fullUrl);
        $decodedPathWithQuery = urldecode($pathWithQuery);

        // Check versions with and without trailing slash
        $pathWithoutSlash = rtrim($path, '/');
        $pathWithSlash = $pathWithoutSlash . '/';
        $decodedPathWithoutSlash = rtrim($decodedPath, '/');
        $decodedPathWithSlash = $decodedPathWithoutSlash . '/';

        // Handle custom redirects
        $redirects = Cache::remember('redirects.map', 300, function () {
            return Redirect::where('is_active', true)->get();
        });

        $match = $redirects->first(function ($r) use (
            $path,
            $fullUrl,
            $decodedPath,
            $decodedFullUrl,
            $pathWithoutSlash,
            $pathWithSlash,
            $decodedPathWithoutSlash,
            $decodedPathWithSlash,
            $pathWithQuery,
            $decodedPathWithQuery
        ) {
            $source = $r->source;
            $decodedSource = urldecode($source);

            // Extract path from source if it's a full URL
            $sourcePath = $source;
            $sourceQuery = '';
            if (filter_var($source, FILTER_VALIDATE_URL)) {
                $sourcePath = parse_url($source, PHP_URL_PATH) ?? '/';
                $sourceQuery = parse_url($source, PHP_URL_QUERY) ?? '';
            } elseif (str_contains($source, '?')) {
                // Handle path with query string
                $parts = explode('?', $source, 2);
                $sourcePath = $parts[0];
                $sourceQuery = $parts[1] ?? '';
            }

            // Build source path with query if exists
            $sourcePathWithQuery = $sourcePath;
            if ($sourceQuery) {
                $sourcePathWithQuery = $sourcePath . '?' . $sourceQuery;
            }

            // Extract path from decoded source if it's a full URL
            $decodedSourcePath = $decodedSource;
            $decodedSourceQuery = '';
            if (filter_var($decodedSource, FILTER_VALIDATE_URL)) {
                $decodedSourcePath = parse_url($decodedSource, PHP_URL_PATH) ?? '/';
                $decodedSourceQuery = parse_url($decodedSource, PHP_URL_QUERY) ?? '';
            } elseif (str_contains($decodedSource, '?')) {
                // Handle path with query string
                $parts = explode('?', $decodedSource, 2);
                $decodedSourcePath = $parts[0];
                $decodedSourceQuery = $parts[1] ?? '';
            }

            // Build decoded source path with query if exists
            $decodedSourcePathWithQuery = $decodedSourcePath;
            if ($decodedSourceQuery) {
                $decodedSourcePathWithQuery = $decodedSourcePath . '?' . $decodedSourceQuery;
            }

            // Create variations with/without trailing slash for paths
            $sourcePathWithoutSlash = rtrim($sourcePath, '/');
            $sourcePathWithSlash = $sourcePathWithoutSlash . '/';
            $decodedSourcePathWithoutSlash = rtrim($decodedSourcePath, '/');
            $decodedSourcePathWithSlash = $decodedSourcePathWithoutSlash . '/';

            // Create variations for original source (in case it's already a path)
            $sourceWithoutSlash = rtrim($source, '/');
            $sourceWithSlash = $sourceWithoutSlash . '/';
            $decodedSourceWithoutSlash = rtrim($decodedSource, '/');
            $decodedSourceWithSlash = $decodedSourceWithoutSlash . '/';

            // Check all possible combinations
            return $source === $path
                || $source === $fullUrl
                || $source === $decodedPath
                || $source === $decodedFullUrl
                || $decodedSource === $path
                || $decodedSource === $decodedPath
                // Path with query string - exact match
                || $source === $pathWithQuery
                || $source === $decodedPathWithQuery
                || $decodedSource === $pathWithQuery
                || $decodedSource === $decodedPathWithQuery
                // Source path with query vs current path with query
                || $sourcePathWithQuery === $pathWithQuery
                || $sourcePathWithQuery === $decodedPathWithQuery
                || $decodedSourcePathWithQuery === $pathWithQuery
                || $decodedSourcePathWithQuery === $decodedPathWithQuery
                // Path extracted from URL variations
                || $sourcePath === $path
                || $sourcePath === $decodedPath
                || $decodedSourcePath === $path
                || $decodedSourcePath === $decodedPath
                // Path with query extracted from full URLs
                || $sourcePath === $pathWithQuery
                || $decodedSourcePath === $pathWithQuery
                // With/without trailing slash variations (original)
                || $sourceWithoutSlash === $pathWithoutSlash
                || $sourceWithSlash === $pathWithSlash
                || $decodedSourceWithoutSlash === $decodedPathWithoutSlash
                || $decodedSourceWithSlash === $decodedPathWithSlash
                // With/without trailing slash variations (extracted paths)
                || $sourcePathWithoutSlash === $pathWithoutSlash
                || $sourcePathWithSlash === $pathWithSlash
                || $decodedSourcePathWithoutSlash === $decodedPathWithoutSlash
                || $decodedSourcePathWithSlash === $decodedPathWithSlash;
        });

        if ($match) {
            $status = (int) $match->status_code;
            if ($status === 410) {
                abort(410);
            }
            if (!empty($match->target)) {
                return redirect()->away($match->target, in_array($status, [301, 302], true) ? $status : 301);
            }
        }

        return $next($request);
    }
}
